package ru.yandex.solomon.util.labelStats;

import java.util.Collection;
import java.util.HashSet;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.Comparators;
import org.apache.commons.lang3.StringUtils;

import ru.yandex.monlib.metrics.labels.validate.LabelValidationFilter;
import ru.yandex.solomon.util.text.TextWithNumbersComparator;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class LabelStats {

    private Set<String> values;
    /**
     * Count metrics with target label. This parameter contains sum of cross dc request, and can't
     * be use to estimate how many metrics matched by particular selector.
     */
    private int count;
    private boolean truncated;

    public LabelStats(Set<String> values, int count, boolean truncated) {
        this.values = values;
        this.count = count;
        this.truncated = truncated;
    }

    public static LabelStats create() {
        return new LabelStats(new HashSet<>(), 0, false);
    }

    public static LabelStats single(String value, int count) {
        Set<String> values = new HashSet<>(1);
        values.add(value);
        return new LabelStats(values, count, false);
    }

    public Set<String> getValues() {
        return values;
    }

    public int getCount() {
        return count;
    }

    public void setCount(int count) {
        this.count = count;
    }

    public boolean isTruncated() {
        return truncated;
    }

    public void setTruncated(boolean truncated) {
        this.truncated = truncated;
    }

    public void add(String value) {
        values.add(value);
        ++count;
    }

    public void addAll(Collection<String> values, int count, boolean truncated) {
        this.values.addAll(values);
        this.count += count;
        this.truncated |= truncated;
    }

    public LabelStats combine(LabelStats target) {
        addAll(target.values, target.count, target.truncated);
        return this;
    }

    public void limit(int count) {
        if (count == 0 || values.size() <= count) {
            return;
        }

        this.values = new HashSet<>(values.stream()
                .collect(Comparators.least(count, TextWithNumbersComparator.instance)));
        this.truncated = true;
    }

    public void filter(String text) {
        if (text.isEmpty()) {
            return;
        }

        this.values = values.stream()
                .parallel()
                .filter(value -> StringUtils.containsIgnoreCase(value, text))
                .collect(Collectors.toSet());
    }

    public void filter(LabelValidationFilter validationFilter) {
        if (validationFilter == LabelValidationFilter.ALL) {
            return;
        }

        this.values = values.stream()
                .parallel()
                .filter(validationFilter::filterValue)
                .collect(Collectors.toSet());
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof LabelStats)) {
            return false;
        }

        LabelStats that = (LabelStats) o;

        if (count != that.count) {
            return false;
        }
        if (truncated != that.truncated) {
            return false;
        }
        return values.equals(that.values);
    }

    @Override
    public int hashCode() {
        int result = values.hashCode();
        result = 31 * result + count;
        result = 31 * result + (truncated ? 1 : 0);
        return result;
    }

    @Override
    public String toString() {
        return "LabelStats{" +
            "values=" + values +
            ", count=" + count +
            ", truncated=" + truncated +
            '}';
    }
}
