package ru.yandex.solomon.util.labelStats;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.function.Predicate;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.lang3.StringUtils;

import ru.yandex.monlib.metrics.labels.Label;
import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.labels.validate.LabelValidationFilter;

/**
 * @author Oleg Baryshnikov
 */
@ParametersAreNonnullByDefault
public class LabelValuesStats {
    private static final Predicate<String> ALWAYS_TRUE = s -> true;

    private Map<String, LabelStats> statsByLabelKey;
    /**
     * Total metrics that was included into stats. This count also include duplicate metrics
     * from different dc, and can't be use to estimate that for particular selector will be
     * return only N metrics.
     */
    private int metricsCount;

    public LabelValuesStats() {
        this.statsByLabelKey = new HashMap<>();
    }

    public LabelValuesStats(Map<String, LabelStats> statsByLabelKey, int metricsCount) {
        this.statsByLabelKey = statsByLabelKey;
        this.metricsCount = metricsCount;
    }

    public Map<String, LabelStats> getStatsByLabelKey() {
        return statsByLabelKey;
    }

    public int getMetricsCount() {
        return metricsCount;
    }

    public void add(Labels labels, Set<String> filter) {
        for (int index = 0; index < labels.size(); index++) {
            Label label = labels.at(index);
            if (filter.isEmpty() || filter.contains(label.getKey())) {
                add(label);
            }
        }
        metricsCount++;
    }

    public void add(Labels labels, Predicate<String> labelPredicate) {
        for (int index = 0; index < labels.size(); index++) {
            Label label = labels.at(index);
            if (labelPredicate.test(label.getKey())) {
                add(label);
            }
        }
        metricsCount++;
    }

    public void add(Label label) {
        getOrCreateStat(label.getKey()).add(label.getValue());
    }

    public void add(String key, LabelStats target) {
        getOrCreateStat(key).combine(target);
    }

    public void replace(String key, LabelStats target) {
        statsByLabelKey.put(key, target);
    }

    public void combine(LabelValuesStats stats) {
        this.metricsCount += stats.metricsCount;
        for (Map.Entry<String, LabelStats> entry : stats.statsByLabelKey.entrySet()) {
            add(entry.getKey(), entry.getValue());
        }
    }

    public void limit(int count) {
        if (count == 0 || metricsCount <= count) {
            return;
        }

        for (LabelStats label : statsByLabelKey.values()) {
            label.limit(count);
        }
    }

    public void limit(int count, Predicate<String> labelPredicate) {
        if (count == 0 || metricsCount <= count) {
            return;
        }

        for (var entry : statsByLabelKey.entrySet()) {
            if (!labelPredicate.test(entry.getKey())) {
                continue;
            }

            entry.getValue().limit(count);
        }
    }

    public void filter(String text) {
        filter(text, ALWAYS_TRUE);
    }

    public void filter(String text, Predicate<String> labelPredicate) {
        if (text.isEmpty()) {
            return;
        }

        for (Map.Entry<String, LabelStats> entry : statsByLabelKey.entrySet()) {
            if (!labelPredicate.test(entry.getKey())) {
                continue;
            }

            if (!StringUtils.containsIgnoreCase(entry.getKey(), text)) {
                entry.getValue().filter(text);
            }
        }
    }

    public void filter(LabelValidationFilter validationFilter) {
        filter(validationFilter, ALWAYS_TRUE);
    }

    public void filter(LabelValidationFilter validationFilter, Predicate<String> labelPredicate) {
        if (validationFilter == LabelValidationFilter.ALL) {
            return;
        }

        for (Map.Entry<String, LabelStats> entry : statsByLabelKey.entrySet()) {
            if (!labelPredicate.test(entry.getKey())) {
                continue;
            }

            LabelStats labelStats = entry.getValue();
            labelStats.filter(validationFilter);
        }
    }

    private LabelStats getOrCreateStat(String key) {
        var stats = statsByLabelKey.get(key);
        if (stats == null) {
            stats = LabelStats.create();
            statsByLabelKey.put(key, stats);
        }
        return stats;
    }

    // It's temporary method to remove metric name stats in MetabaseShard#labelStatsInNewFormat
    public void removeByKey(String oldKey) {
        statsByLabelKey.remove(oldKey);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof LabelValuesStats)) {
            return false;
        }

        LabelValuesStats stats = (LabelValuesStats) o;

        if (metricsCount != stats.metricsCount) {
            return false;
        }
        return statsByLabelKey.equals(stats.statsByLabelKey);
    }

    @Override
    public int hashCode() {
        int result = statsByLabelKey.hashCode();
        result = 31 * result + metricsCount;
        return result;
    }

    @Override
    public String toString() {
        return "LabelValuesStats{" +
            "statsByLabelKey=" + statsByLabelKey +
            ", sensorsCount=" + metricsCount +
            '}';
    }
}
