package ru.yandex.solomon.util.net;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.math.BigInteger;
import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.stream.Collectors;

import com.github.jgonian.ipmath.Ipv4Range;
import com.github.jgonian.ipmath.Ipv6Range;

import ru.yandex.misc.ExceptionUtils;

/**
 * @author alexlovkov
 */
public class DefaultNetworkValidator implements NetworkValidator {

    private final List<Ipv4Range> ipv4Addresses;
    private final List<Ipv6Range> ipv6Addresses;
    private static final BigInteger maxNetSize = BigInteger.valueOf(1024);

    public DefaultNetworkValidator(List<String> ipv4Nets, List<String> ipv6Nets) {
        this.ipv4Addresses = ipv4Nets.stream().map(Ipv4Range::parse).collect(Collectors.toList());
        this.ipv6Addresses = ipv6Nets.stream().map(Ipv6Range::parse).collect(Collectors.toList());
    }

    @Override
    public void validateNetwork(String network) {
        BigInteger networkSize = NetworkUtils.getNetworkSize(network);
        if (networkSize.compareTo(maxNetSize) > 0) {
            throw new IllegalArgumentException("too many addresses in network:" + network);
        }
        NetworkUtils.insideYandexNetworks(network, ipv4Addresses, ipv6Addresses);
    }

    private static List<String> loadFromStream(InputStream stream) {
        BufferedReader reader = new BufferedReader(new InputStreamReader(stream, StandardCharsets.UTF_8));
        List<String> strings = reader.lines().collect(Collectors.toList());
        if (strings.isEmpty()) {
            throw new RuntimeException("file with yandex networks is empty");
        }
        return strings;
    }

    public static NetworkValidator loadFromClassPath() {
        List<String> ipv4Strings;
        List<String> ipv6Strings;
        try (InputStream inputStream = DefaultNetworkValidator.class.getResourceAsStream("yandex_ipv4_nets.txt")) {
            ipv4Strings = loadFromStream(inputStream);
        } catch (IOException e) {
            throw ExceptionUtils.throwException(e);
        }
        try (InputStream inputStream = DefaultNetworkValidator.class.getResourceAsStream("yandex_ipv6_nets.txt")) {
            ipv6Strings = loadFromStream(inputStream);
        } catch (IOException e) {
            throw ExceptionUtils.throwException(e);
        }
        return new DefaultNetworkValidator(ipv4Strings, ipv6Strings);
    }
}
