package ru.yandex.solomon.util.net;

import java.math.BigInteger;
import java.util.List;

import com.github.jgonian.ipmath.Ipv4;
import com.github.jgonian.ipmath.Ipv4Range;
import com.github.jgonian.ipmath.Ipv6;
import com.github.jgonian.ipmath.Ipv6Range;

/**
 * @author alexlovkov
 */
public class NetworkUtils {

    private static final BigInteger TWO = BigInteger.valueOf(2);

    public static BigInteger getNetworkSize(String network) {
        if (network.contains("/")) {
            try {
                Ipv4Range ipv4Range = Ipv4Range.parse(network);
                return BigInteger.valueOf(ipv4Range.size() - 2);
            } catch (IllegalArgumentException e) {
                Ipv6Range ipv6Range = Ipv6Range.parse(network);
                return ipv6Range.size().subtract(TWO);
            }
        }
        try {
            Ipv4.parse(network);
            return BigInteger.ONE;
        } catch (IllegalArgumentException ignore) {}

        try {
            Ipv6.parse(network);
            return BigInteger.ONE;
        } catch (IllegalArgumentException ignore) {}

        throw new IllegalArgumentException("network:" + network + " is not correct");
    }

    public static void insideYandexNetworks(
        String address,
        List<Ipv4Range> ipv4Addresses,
        List<Ipv6Range> ipv6Addresses)
    {
        RuntimeException exception = null;
        if (address.contains("/")) {
            try {
                Ipv4Range ipv4 = Ipv4Range.parse(address);
                if (!insideYandexNetwork(ipv4, ipv4Addresses)) {
                    exception = new IllegalArgumentException("ipv4 " + address + " is not inside yandex networks");
                }
            } catch (Exception e) {
                Ipv6Range ipv6 = Ipv6Range.parse(address);
                if (!insideYandexNetwork(ipv6, ipv6Addresses)) {
                    exception = new IllegalArgumentException("ipv6 " + address + " is not inside yandex networks");
                }
            }
        } else {
            try {
                Ipv4 ipv4 = Ipv4.parse(address);
                Ipv4Range range = Ipv4Range.from(ipv4).to(ipv4);
                if (!insideYandexNetwork(range, ipv4Addresses)) {
                    exception = new IllegalArgumentException("ipv4 " + address + " is not inside yandex networks");
                }
            } catch (Exception e) {
                Ipv6 ipv6 = Ipv6.parse(address);
                Ipv6Range range = Ipv6Range.from(ipv6).to(ipv6);
                if (!insideYandexNetwork(range, ipv6Addresses)) {
                    exception = new IllegalArgumentException("ipv6 " + address + " is not inside yandex networks");
                }
            }
        }
        if (exception != null) {
            throw exception;
        }
    }

    private static boolean insideYandexNetwork(
        Ipv4Range address,
        List<Ipv4Range> ipv4Addresses)
    {
        for (Ipv4Range current : ipv4Addresses) {
            // user try to use first or last ip in the net
            if (address.start().equals(address.end()) &&
                (address.start().equals(current.start()) || address.start().equals(current.end()))) {
                continue;
            }
            if (current.contains(address)) {
                return true;
            }
        }
        return false;
    }

    private static boolean insideYandexNetwork(
        Ipv6Range address,
        List<Ipv6Range> ipv6Addresses)
    {
        for (Ipv6Range current : ipv6Addresses) {
            // user try to use first or last ip in the net
            if (address.start().equals(address.end()) &&
                (address.start().equals(current.start()) || address.start().equals(current.end()))) {
                continue;
            }
            if (current.contains(address)) {
                return true;
            }
        }
        return false;
    }
}
