package ru.yandex.solomon.util.parser;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.function.Predicate;
import java.util.function.Supplier;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.annotation.ParametersAreNonnullByDefault;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class ParserSupport {
    protected final String string;
    protected int pos;

    public ParserSupport(String string) {
        this(string, 0);
    }

    public ParserSupport(String string, int pos) {
        this.string = string;
        this.pos = pos;
    }

    public boolean hasNext() {
        return pos < string.length();
    }

    public int getPos() {
        return pos;
    }

    public char lookahead() {
        return string.charAt(pos);
    }

    public String lookaheadString() {
        return string.substring(pos);
    }

    public boolean lookaheadIs(char c) {
        return hasNext() && lookahead() == c;
    }

    public boolean lookaheadIs(Predicate<Character> p) {
        return hasNext() && p.test(lookahead());
    }

    public boolean lookaheadIs(String s) {
        return lookaheadString().startsWith(s);
    }

    public ParserSupport copy() {
        return new ParserSupport(string, pos);
    }

    public void copyTo(ParserSupport other) {
        other.pos = pos;
    }

    public char consumeChar() {
        return string.charAt(pos++);
    }

    public void consume(char c) {
        if (!lookaheadIs(c)) {
            throw new RuntimeException("expecting: '" + c + "' after pos " + pos + " for [" + string + "]");
        }
        consumeChar();
    }

    public String consume(String s) {
        if (!consumeOptional(s)) {
            throw new RuntimeException("expecting: '" + s + "' after pos " + pos + " for [" + string + "]");
        }
        return s;
    }

    public Optional<String> consumeOptionalOneOf(String... ss) {
        for (String s : ss) {
            if (consumeOptional(s)) {
                return Optional.of(s);
            }
        }
        return Optional.empty();
    }

    public Optional<Matcher> consumeOptional2(Pattern pattern) {
        Matcher matcher = pattern.matcher(lookaheadString());
        if (!matcher.lookingAt()) {
            return Optional.empty();
        }
        consume(matcher.group());
        return Optional.of(matcher);
    }

    public Optional<String> consumeOptional(Pattern pattern) {
        return consumeOptional2(pattern).map(Matcher::group);
    }

    public String consume(Pattern pattern) {
        return consumeOptional(pattern).orElseThrow(
                () -> new RuntimeException("expecting: " + pattern + ", ahead: " + lookaheadString() + ", string: " + string));
    }

    public String nextStringUpToSep(char sep) {
        StringBuilder sb = new StringBuilder();
        while (hasNext() && lookahead() != sep) {
            sb.append(consumeChar());
        }
        return sb.toString();
    }

    public boolean lookingAtEof() {
        return !hasNext();
    }

    public void checkEof() {
        if (!lookingAtEof()) {
            throw new RuntimeException("EOF is reached");
        }
    }

    public int nextDecimalDigit() {
        if (!lookaheadIsDecimal()) {
            throw new RuntimeException("lookahead is not decimal; pos: " + pos + "; string: " + string);
        }
        return consumeChar() - '0';
    }

    public boolean lookaheadIsDecimal() {
        return hasNext() && lookahead() >= '0' && lookahead() <= '9';
    }

    public long nextUnsignedLongDecimal() {
        long r = nextDecimalDigit();
        while (lookaheadIsDecimal()) {
            r = r * 10 + nextDecimalDigit();
        }
        return r;
    }

    public int nextUnsignedIntDecimal() {
        long r = nextUnsignedLongDecimal();
        if (r > Integer.MAX_VALUE) {
            throw new RuntimeException("too much: " + r);
        }
        return (int) r;
    }

    public long nextSignedLongDecimal() {
        boolean minus = consumeOptional('-');
        long unsigned = nextUnsignedLongDecimal();
        return minus ? -unsigned : unsigned;
    }

    public int nextSignedIntDecimal() {
        boolean minus = consumeOptional('-');
        int unsigned = nextUnsignedIntDecimal();
        return minus ? -unsigned : unsigned;
    }

    public boolean consumeOptional(char c) {
        if (lookaheadIs(c)) {
            consume(c);
            return true;
        } else {
            return false;
        }
    }

    public boolean consumeOptional(String s) {
        if (lookaheadIs(s)) {
            pos += s.length();
            return true;
        } else {
            return false;
        }
    }

    public String consumeRest() {
        String r = lookaheadString();
        consume(r);
        checkEof();
        return r;
    }

    public String consumeUntil(Pattern pattern) {
        Matcher matcher = pattern.matcher(lookaheadString());
        if (matcher.find()) {
            return consumeN(matcher.start());
        } else {
            return consumeRest();
        }
    }

    public String consumeUntil(String string) {
        int where = lookaheadString().indexOf(string);
        if (where >= 0) {
            return consumeN(where);
        } else {
            return consumeRest();
        }
    }

    public String consumeWhile(Predicate<Character> p) {
        StringBuilder r = new StringBuilder();
        while (lookaheadIs(p)) {
            r.append(consumeChar());
        }
        return r.toString();
    }

    public String consumeWhitespaces() {
        return consumeWhile(Character::isWhitespace);
    }

    public String consumeUntil(char c) {
        return consumeWhile(n -> n != c);
    }

    public String consumeN(int count) {
        String r = lookaheadString().substring(0, count);
        return consume(r);
    }

    public <T> List<T> parseSeq(Supplier<T> itemParser) {
        List<T> r = new ArrayList<>();
        while (!lookingAtEof()) {
            r.add(itemParser.get());
        }
        return r;
    }
}
