package ru.yandex.solomon.util.protobuf;

import java.lang.invoke.MethodHandle;
import java.lang.invoke.MethodHandles;
import java.lang.reflect.Field;

import com.google.protobuf.ByteString;
import com.google.protobuf.UnsafeByteOperations;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufUtil;
import io.netty.buffer.Unpooled;
import it.unimi.dsi.fastutil.bytes.ByteArrays;

import ru.yandex.solomon.util.ExceptionUtils;


/**
 * @author Sergey Polovko
 */
public final class ByteStrings {
    private ByteStrings() {}

    private static final Class<?> LITERAL_BYTE_STRING_CLASS;
    private static MethodHandle BYTES_GETTER;

    // ByteString trying to be immutable and has no method to
    // get underlying byte[] array without copying it, so we
    // use reflection to get around this boundaries
    static {
        ByteString str = ByteString.copyFrom(new byte[0]);
        LITERAL_BYTE_STRING_CLASS = str.getClass();
        try {
            Field bytesField = LITERAL_BYTE_STRING_CLASS.getDeclaredField("bytes");
            bytesField.setAccessible(true);
            BYTES_GETTER = MethodHandles.lookup().unreflectGetter(bytesField);
        } catch (Throwable t) {
            ExceptionUtils.uncaughtException(t);
        }
    }

    public static ByteBuf toByteBuf(ByteString str) {
        if (str.isEmpty()) {
            return Unpooled.EMPTY_BUFFER;
        }

        if (LITERAL_BYTE_STRING_CLASS == str.getClass()) {
            try {
                return Unpooled.wrappedBuffer((byte[]) BYTES_GETTER.invoke(str));
            } catch (Throwable t) {
                throw new RuntimeException(t);
            }
        }

        // slow pass
        return Unpooled.wrappedBuffer(str.toByteArray());
    }

    public static byte[] toByteArray(ByteString str) {
        if (str.isEmpty()) {
            return ByteArrays.EMPTY_ARRAY;
        }

        if (LITERAL_BYTE_STRING_CLASS == str.getClass()) {
            try {
                return (byte[]) BYTES_GETTER.invoke(str);
            } catch (Throwable t) {
                throw new RuntimeException(t);
            }
        }

        // slow pass
        return str.toByteArray();
    }

    public static ByteString fromByteBuf(ByteBuf buf) {
        if (!buf.isReadable()) {
            return ByteString.EMPTY;
        }
        if (buf.isDirect()) {
            return UnsafeByteOperations.unsafeWrap(buf.nioBuffer());
        }
        if (buf.hasArray()) {
            byte[] array = buf.array();
            int offset = buf.arrayOffset() + buf.readerIndex();
            int length = buf.readableBytes();
            return UnsafeByteOperations.unsafeWrap(array, offset, length);
        }
        // slow pass
        return UnsafeByteOperations.unsafeWrap(ByteBufUtil.getBytes(buf));
    }

    public static ByteString[] split(ByteString bytes, int maxSize) {
        if (bytes.isEmpty()) {
            return new ByteString[0];
        }

        if (bytes.size() <= maxSize) {
            return new ByteString[]{ bytes };
        }

        int partCount = (bytes.size() + maxSize - 1) / maxSize;
        ByteString[] result = new ByteString[partCount];
        for (int i = 0; i < partCount; i++) {
            int from = i * maxSize;
            int to = from + maxSize;
            result[i] = bytes.substring(from, Math.min(bytes.size(), to));
        }
        return result;
    }
}
