package ru.yandex.solomon.util.protobuf;

import com.google.protobuf.ByteString;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.Unpooled;
import net.jpountz.lz4.LZ4Factory;

/**
 * @author Sergey Polovko
 */
final class Lz4Codec {

    static ByteString encode(ByteBuf buffer) {
        var compressor = LZ4Factory.fastestInstance().fastCompressor();

        byte[] src = buffer.array();
        int srcLen = buffer.readableBytes();
        int srcOff = buffer.arrayOffset() + buffer.readerIndex();

        int maxDstLen = compressor.maxCompressedLength(srcLen);
        byte[] dst = new byte[maxDstLen];

        int compressedLen = compressor.compress(src, srcOff, srcLen, dst, 0, maxDstLen);
        return ByteString.copyFrom(dst, 0, compressedLen);
    }

    static ByteBuf decode(ByteString compressed, int originalSize) {
        ByteBuf uncompressed = Unpooled.buffer(originalSize);
        byte[] bytes = ByteStrings.toByteArray(compressed);

        var decompressor = LZ4Factory.fastestInstance().fastDecompressor();
        int readBytes = decompressor.decompress(bytes, 0, uncompressed.array(), 0, originalSize);
        if (readBytes != compressed.size()) {
            throw new RuntimeException(String.format(
                    "lz4 stream is corrupted: readBytes(%d) != compressedSize(%d)",
                    readBytes, compressed.size()));
        }

        uncompressed.writerIndex(originalSize);
        return uncompressed;
    }
}
