package ru.yandex.solomon.util.protobuf;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import com.google.common.io.Files;
import com.google.protobuf.Message;
import com.google.protobuf.MessageOrBuilder;
import com.google.protobuf.TextFormat;

import ru.yandex.misc.ExceptionUtils;

/**
 * @author Stepan Koltsov
 */
public class ProtobufText {

    public static <M extends Message> M readFromTextFile(File file, M defaultInstance) {
        try (BufferedReader reader = Files.newReader(file, StandardCharsets.UTF_8)) {
            Message.Builder builder = defaultInstance.newBuilderForType();
            TextFormat.merge(reader, builder);
            return (M) builder.build();
        } catch (IOException e) {
            throw ExceptionUtils.throwException(e);
        }
    }

    public static void writeToTextFile(File file, MessageOrBuilder message) {
        try (BufferedWriter writer = Files.newWriter(file, StandardCharsets.UTF_8)) {
            TextFormat.printUnicode(message, writer);
        } catch (IOException e) {
            throw ExceptionUtils.throwException(e);
        }
    }

    public static String serializeToText(MessageOrBuilder message) {
        return TextFormat.printToUnicodeString(message);
    }

    public static String serializeToTextEnsureTrailingLf(MessageOrBuilder message) {
        String r = serializeToText(message);
        if (!r.isEmpty() && !r.endsWith("\n")) {
            r += "\n";
        }
        return r;
    }

    public static <M extends Message> M readFromStringThrowSafe(String prototext, M defaultInstance) {
        Message.Builder builder = defaultInstance.newBuilderForType();
        try {
            TextFormat.merge(prototext, builder);
            return (M) builder.build();
        } catch (TextFormat.ParseException e) {
            return defaultInstance;
        }
    }

    public static <M extends Message> M readFromString(String prototext, M defaultInstance) {
        Message.Builder builder = defaultInstance.newBuilderForType();
        try {
            TextFormat.merge(prototext, builder);
            return (M) builder.build();
        } catch (TextFormat.ParseException e) {
            throw new RuntimeException(e);
        }
    }

    public static <M extends Message> List<M> readFromStrings(M defaultInstance, String... prototexts) {
        return Stream.of(prototexts)
                .map(s -> readFromString(s, defaultInstance))
                .collect(Collectors.toList());
    }
}
