package ru.yandex.solomon.util.protobuf;

import java.util.Arrays;

import com.google.protobuf.ByteString;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.Unpooled;
import io.netty.util.CharsetUtil;
import it.unimi.dsi.fastutil.objects.Object2IntOpenHashMap;

import ru.yandex.solomon.common.StringPool.Compression;
import ru.yandex.solomon.util.collection.array.ArrayBuilder;

/**
 * @author Sergey Polovko
 */
public class StringPool {

    private final String[] strings;

    public StringPool(String... strings) {
        this.strings = strings;
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public static StringPool fromProto(ru.yandex.solomon.common.StringPool proto) {
        ByteBuf decompressed = decompress(proto.getCompression(), proto.getStrings(), proto.getOriginalSize());

        try {
            var builder = new ArrayBuilder<>(String.class);
            // add an empty string at index 0, to make sure that with default initialized ID
            // we will return default initialized (an empty) string
            builder.add("");

            while (decompressed.isReadable()) {
                int len = decompressed.bytesBefore((byte) 0);
                CharSequence value = decompressed.readCharSequence(len, CharsetUtil.UTF_8);
                decompressed.readByte(); // skip delimiter

                // TODO: remove this temporary hack after fixes from https://a.yandex-team.ru/review/1820040/details
                //       will be deployed to all DataProxy clients in production
                if (!value.isEmpty()) {
                    builder.add(value.toString());
                }
            }

            return new StringPool(builder.buildUnsafe());
        } finally {
            decompressed.release();
        }
    }

    public String get(int index) {
        return strings[index];
    }

    public int size() {
        return strings.length - 1; // -1 for an empty string
    }

    @Override
    public String toString() {
        return "StringPool" + Arrays.toString(strings);
    }

    private static ByteBuf decompress(Compression compression, ByteString buffer, int originalSize) {
        switch (compression) {
            case COMPRESSION_UNSPECIFIED: return ByteStrings.toByteBuf(buffer);
            case LZ4: return Lz4Codec.decode(buffer, originalSize);
            default:
                throw new IllegalArgumentException("unsupported compression: " + compression);
        }
    }

    private static ByteString compress(Compression compression, ByteBuf buffer) {
        switch (compression) {
            case COMPRESSION_UNSPECIFIED: return ByteStrings.fromByteBuf(buffer);
            case LZ4: return Lz4Codec.encode(buffer);
            default:
                throw new IllegalArgumentException("unsupported compression: " + compression);
        }
    }

    /**
     * BUILDER
     */
    public static final class Builder {
        private final Object2IntOpenHashMap<String> index;
        private final ArrayBuilder<String> strings;
        private int counter = 0;
        private int expectedSize = 0;

        public Builder() {
            this.index = new Object2IntOpenHashMap<>();
            this.index.defaultReturnValue(-1);
            this.strings = new ArrayBuilder<>(String.class);

            // add an empty string at index 0, to make sure that with default initialized ID
            // we will return default initialized (an empty) string
            strings.add("");
        }

        public int put(String value) {
            if (value == null || value.isEmpty()) {
                return 0;
            }

            int idx = index.getInt(value);
            if (idx != -1) {
                return idx;
            }

            idx = ++counter;
            index.put(value, idx);
            strings.add(value);
            expectedSize += value.length();
            return idx;
        }

        public int size() {
            return strings.size - 1; // -1 for an empty string
        }

        public StringPool build() {
            return new StringPool(strings.buildUnsafe());
        }

        public ru.yandex.solomon.common.StringPool buildProto(Compression compression) {
            if (index.isEmpty()) {
                return ru.yandex.solomon.common.StringPool.getDefaultInstance();
            }

            var buffer = Unpooled.buffer(expectedSize + index.size());
            try {
                // TODO: remove this temporary hack (i=0) after fixes from https://a.yandex-team.ru/review/1820040/details
                //       will be deployed to all DataProxy clients in production
                for (int i = 0; i < strings.size; i++) {
                    buffer.writeCharSequence(strings.array[i], CharsetUtil.UTF_8);
                    buffer.writeByte(0); // delimiter
                }

                return ru.yandex.solomon.common.StringPool.newBuilder()
                        .setCompression(compression)
                        .setOriginalSize(buffer.readableBytes())
                        .setStrings(compress(compression, buffer))
                        .build();
            } finally {
                buffer.release();
            }
        }
    }
}
