package ru.yandex.solomon.util.time;

import java.time.Duration;
import java.time.Instant;
import java.time.temporal.ChronoUnit;

import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.concurrent.Immutable;
import javax.annotation.concurrent.ThreadSafe;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.bolts.collection.Option;
import ru.yandex.misc.thread.ThreadLocalTimeout;
import ru.yandex.misc.thread.ThreadLocalTimeoutException;

/**
 * @author chebik
 */
@ThreadSafe
@Immutable
@ParametersAreNonnullByDefault
public class Deadline {

    public static final Duration DEFAULT_TIMEOUT = Duration.ofSeconds(30);
    public static final long DEFAULT_TIMEOUT_MILLIS = DEFAULT_TIMEOUT.toMillis();

    private static final Logger logger = LoggerFactory.getLogger(Deadline.class);

    private final String opName;
    private final Instant deadline;

    private Deadline(String opName, Instant deadline) {
        this.opName = opName;
        this.deadline = deadline;
    }

    public void check() {
        Instant now = Instant.now();
        if (now.isAfter(deadline)) {
            long millisSinceDeadline = deadline.until(now, ChronoUnit.MILLIS);
            throw new ThreadLocalTimeoutException(
                "Operation " + opName + " timed out. Passed " + (millisSinceDeadline / 1000.0) +
                    " seconds since deadline, deadline was " + deadline);
        }
    }

    public long toEpochMilli() {
        return deadline.toEpochMilli();
    }

    public Instant toInstant() {
        return deadline;
    }

    public org.joda.time.Instant toJodaInstant() {
        return new org.joda.time.Instant(deadline.toEpochMilli());
    }

    public static Deadline fromThreadLocal(String opName) {
        Option<org.joda.time.Instant> deadlineOption = ThreadLocalTimeout.deadline();
        if (deadlineOption.isPresent()) {
            return new Deadline(opName, Instant.ofEpochMilli(deadlineOption.get().getMillis()));
        } else {
            logger.warn("No timeout set in thread local for " + opName + ", falling back to default timeout (" + DEFAULT_TIMEOUT + ")");
            return defaultDeadline(opName);
        }
    }

    private static Deadline defaultDeadline(String opName) {
        return new Deadline(opName, Instant.now().plus(DEFAULT_TIMEOUT));
    }

    public static Deadline from(Instant time) {
        return from("", time);
    }

    public static Deadline from(String opName, Instant time) {
        return new Deadline(opName, time);
    }

    public static Deadline within(Duration duration) {
        return new Deadline("", Instant.now().plus(duration));
    }

    public static Deadline withinMinute() {
        return within(Duration.ofMinutes(1));
    }

    public static Instant defaultDeadline() {
        return Instant.now().plusMillis(Deadline.DEFAULT_TIMEOUT_MILLIS);
    }
}
