package ru.yandex.solomon.util.time;

import java.time.Duration;
import java.util.Arrays;
import java.util.Locale;
import java.util.Optional;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class DurationUtils {

    private static final int SECONDS_PER_MINUTE = 60;
    private static final int SECONDS_PER_HOUR = SECONDS_PER_MINUTE * 60;
    private static final int SECONDS_PER_DAY = SECONDS_PER_HOUR * 24;
    private static final int SECONDS_PER_WEEK = SECONDS_PER_DAY * 7;

    private static final Pattern durationElemPattern = Pattern.compile("^((\\d+)(w|d|h|m|s)).*");
    private static final TimeUnit[] TIME_UNITS = {TimeUnit.DAYS, TimeUnit.HOURS, TimeUnit.MINUTES, TimeUnit.SECONDS};

    // XXX: copy-pasted from GggDurationSerializer
    public static Optional<Duration> parseDuration(@Nullable String duration) {
        if (duration == null || duration.isEmpty()) {
            return Optional.empty();
        }

        String unparsedDuration = duration.replaceFirst("^-", "");

        if (unparsedDuration.equals("0")) {
            return Optional.of(Duration.ZERO);
        }

        int sumInSeconds = 0;
        while (unparsedDuration.length() > 0) {
            Matcher matchResult = durationElemPattern.matcher(unparsedDuration);
            if (!matchResult.matches()) {
                return Optional.empty();
            }

            int count = Integer.parseInt(matchResult.group(2));
            switch (matchResult.group(3).charAt(0)) {
                case 'w':
                    sumInSeconds += count * SECONDS_PER_WEEK;
                    break;
                case 'd':
                    sumInSeconds += count * SECONDS_PER_DAY;
                    break;
                case 'h':
                    sumInSeconds += count * SECONDS_PER_HOUR;
                    break;
                case 'm':
                    sumInSeconds += count * SECONDS_PER_MINUTE;
                    break;
                case 's':
                    sumInSeconds += count;
                    break;
                default:
                    return Optional.empty();
            }

            unparsedDuration = unparsedDuration.substring(matchResult.group(1).length());
        }

        return Optional.of(Duration.ofSeconds(sumInSeconds));
    }

    public static long randomize(long time) {
        if (time == 0) {
            return 0;
        }

        var half = Math.max(time / 2, 1);
        return half + ThreadLocalRandom.current().nextLong(half);
    }

    public static long backoff(long delay, long maxDelay, int attempt) {
        return Math.min(Math.round(delay * Math.pow(2, attempt)), maxDelay);
    }

    public static String toStringUpToMillis(Duration d) {
        if (d.isNegative()) {
            return "-" + toStringUpToMillis(d.negated());
        }
        return String.format(Locale.US, "%d.%03d", d.getSeconds(), d.getNano() / 1000000);
    }

    public static String millisToSecondsString(long millis) {
        return toStringUpToMillis(Duration.ofMillis(millis));
    }

    public static String millisToSecondsStringToNow(long start) {
        return millisToSecondsString(System.currentTimeMillis() - start);
    }

    public static long secondsToMillis(long seconds) {
        return seconds * 1000;
    }

    public static long millisToSeconds(long millis) {
        return millis / 1000;
    }

    public static int millisToSeconds(int millis) {
        return millis / 1000;
    }

    public static String formatDurationMillis(long millis) {
        return formatDurationSeconds(millis / 1000);
    }

    public static String formatDurationMillisTruncated(long millis) {
        boolean isNegative = millis < 0;
        millis = Math.abs(millis);
        for (TimeUnit unit : Arrays.asList(TimeUnit.DAYS, TimeUnit.HOURS, TimeUnit.MINUTES, TimeUnit.SECONDS)) {
            long mod = unit.toMillis(1);
            if (millis > mod) {
                millis -= millis % mod;
                break;
            }
        }

        String result = DurationUtils.formatDurationMillis(millis);
        if (isNegative) {
            result = "-" + result;
        }
        return result;
    }

    public static String formatDurationSeconds(long seconds) {
        long days = seconds / SECONDS_PER_DAY;
        seconds -= days * SECONDS_PER_DAY;

        long hours = seconds / SECONDS_PER_HOUR;
        seconds -= hours * SECONDS_PER_HOUR;

        long minutes = seconds / SECONDS_PER_MINUTE;
        seconds -= minutes * SECONDS_PER_MINUTE;

        StringBuilder sb = new StringBuilder();
        if (days != 0) {
            sb.append(days).append('d');
        }
        if (hours != 0) {
            sb.append(hours).append('h');
        }
        if (minutes != 0) {
            sb.append(minutes).append('m');
        }
        if (seconds != 0) {
            sb.append(seconds).append('s');
        }

        if (sb.length() == 0) {
            return "0";
        }

        return sb.toString();
    }

    public static String formatDurationNanos(long nanos) {
        if (nanos == 0) {
            return "0";
        }

        long millis = TimeUnit.NANOSECONDS.toMillis(nanos);
        if (millis == 0) {
            return nanos + " ns";
        }

        if (millis < 1000) {
            return millis + " ms";
        }

        return DurationUtils.formatDurationMillis(millis);
    }

    public static String formatDuration(Duration duration) {
        if (duration.isNegative()) {
            return "-" + formatDuration(duration.negated());
        }
        return formatDurationMillis(duration.toMillis());
    }

    public static String toHumanReadable(long millis) {
        boolean isNegative = millis < 0;
        millis = Math.abs(millis);
        for (TimeUnit unit : TIME_UNITS) {
            long mod = unit.toMillis(1);
            if (millis > mod) {
                millis -= millis % mod;
                break;
            }
        }

        String result = DurationUtils.formatDurationMillis(millis);
        if (isNegative) {
            result = "-" + result;
        }
        return result;
    }
}
