package ru.yandex.solomon.util.time;

import java.time.Instant;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeFormatterBuilder;
import java.time.temporal.ChronoField;
import java.util.Locale;
import java.util.concurrent.TimeUnit;

import static java.time.temporal.ChronoField.INSTANT_SECONDS;
import static java.time.temporal.ChronoField.NANO_OF_SECOND;

/**
 * @author Stepan Koltsov
 */
public class InstantUtils {

    public static final DateTimeFormatter toMillisFormatter = new DateTimeFormatterBuilder()
            .appendInstant(3)
            .toFormatter(Locale.US);

    public static final DateTimeFormatter toSecondsFormatter = new DateTimeFormatterBuilder()
            .appendInstant(0)
            .toFormatter(Locale.US);

    public static String formatToMillis(long instant) {
        return toMillisFormatter.format(Instant.ofEpochMilli(instant));
    }

    public static String formatToSeconds(long instantMillis) {
        return toSecondsFormatter.format(Instant.ofEpochMilli(instantMillis));
    }

    public static String millisecondsToSecondsString(long instant) {
        return String.format(Locale.US, "%d.%03d", instant / 1000, instant % 1000);
    }


    public static final long NOT_BEFORE = Instant.parse("2000-01-01T00:00:00Z").toEpochMilli();
    public static final long NOT_AFTER = Instant.parse("2038-01-19T03:14:07Z").toEpochMilli();
    static {
        // sanity check
        millisecondsToSeconds(NOT_AFTER);
    }

    public static boolean isGoodMillis(long millis) {
        return millis >= NOT_BEFORE && millis <= NOT_AFTER;
    }

    public static boolean checkGridMultiplicity(long left, long right) {
        if (left > right) {
            return (left % right) == 0;
        }
        return (right % left) == 0;
    }

    public static boolean isGoodSeconds(long seconds) {
        return isGoodMillis(seconds * 1000);
    }

    public static void sanityCheckMillis(long millis) {
        if (!isGoodMillis(millis)) {
            throw new RuntimeException("does not satisfy sanity check: " + millis);
        }
    }

    public static void sanityCheckSeconds(long instantSeconds) {
        sanityCheckMillis(instantSeconds * 1000);
    }

    public static int millisecondsToSeconds(long millis) {
        return Math.toIntExact(millis / 1000);
    }

    public static long secondsToMillis(long seconds) {
        return seconds * 1000;
    }

    public static int currentTimeSeconds() {
        return millisecondsToSeconds(System.currentTimeMillis());
    }

    public static long parseToMillis(String text) {
        return DateTimeFormatter.ISO_INSTANT.parse(text, temporal -> {
            long secs = temporal.getLong(INSTANT_SECONDS);
            long nanoOfSecond = temporal.get(NANO_OF_SECOND);
            return TimeUnit.SECONDS.toMillis(secs) + TimeUnit.NANOSECONDS.toMillis(nanoOfSecond);
        });
    }

    public static int parseToSeconds(String text) {
        return DateTimeFormatter.ISO_INSTANT.parse(text,
            temporal -> Math.toIntExact(temporal.getLong(ChronoField.INSTANT_SECONDS)));
    }

    public static Instant truncate(Instant time, long gridMillis) {
        return Instant.ofEpochMilli(truncate(time.toEpochMilli(), gridMillis));
    }

    public static long truncate(long timeMillis, long gridMillis) {
        long diff = timeMillis % gridMillis;
        return timeMillis - diff;
    }

    public static Instant ceil(Instant time, long gridMillis) {
        return Instant.ofEpochMilli(ceil(time.toEpochMilli(), gridMillis));
    }

    public static long ceil(long timeMillis, long gridMillis) {
        long diff = timeMillis % gridMillis;
        return diff == 0 ? timeMillis : timeMillis + gridMillis - diff;
    }

    public static void main(String[] args) {
        long now = System.currentTimeMillis();
        System.out.println("             formatToMillis: " + formatToMillis(now));
        System.out.println("            formatToSeconds: " + formatToSeconds(now));
        System.out.println("millisecondsToSecondsString: " + millisecondsToSecondsString(now));
    }
}
