package ru.yandex.solomon.util.time;

import java.time.Duration;
import java.time.Instant;
import java.time.ZoneId;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

/**
 * @author Stepan Koltsov
 */
@ParametersAreNonnullByDefault
public class Interval {
    public static final Interval EMPTY = new Interval(Instant.EPOCH, Instant.EPOCH);

    @Nonnull
    private final Instant begin;
    @Nonnull
    private final Instant end;

    public Interval(Instant begin, Instant end) {
        if (begin.compareTo(end) > 0) {
            throw new IllegalArgumentException("begin is le than end: " + begin + ", " + end);
        }
        this.begin = begin;
        this.end = end;
    }

    public static Interval before(Instant end, Duration duration) {
        return new Interval(end.minus(duration), end);
    }

    public static Interval after(Instant begin, Duration duration) {
        return new Interval(begin, begin.plus(duration));
    }

    @Nonnull
    public Instant getBegin() {
        return begin;
    }

    @Nonnull
    public Instant getEnd() {
        return end;
    }

    public long getBeginSeconds() {
        return begin.getEpochSecond();
    }

    public long getEndSeconds() {
        return end.getEpochSecond();
    }

    public long getBeginMillis() {
        return begin.toEpochMilli();
    }

    public long getEndMillis() {
        return end.toEpochMilli();
    }

    public Duration duration() {
        return Duration.between(begin, end);
    }

    public LocalDateTimeInterval toLocalDateTimeInterval(ZoneId zoneId) {
        return new LocalDateTimeInterval(begin.atZone(zoneId).toLocalDateTime(), end.atZone(zoneId).toLocalDateTime());
    }

    public ZonedDateTimeInterval atZone(ZoneId zoneId) {
        return new ZonedDateTimeInterval(toLocalDateTimeInterval(zoneId), zoneId);
    }

    public Duration length() {
        return Duration.between(begin, end);
    }

    public boolean containsOpenClose(Instant instant) {
        if (instant.isBefore(begin)) {
            return false;
        }
        if (!instant.isBefore(end)) {
            return false;
        }
        return true;
    }

    @Override
    public String toString() {
        return "[" + begin + "; " + end + "]";
    }

    @Override
    public boolean equals(@Nullable Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        Interval interval = (Interval) o;

        if (!begin.equals(interval.begin)) return false;
        if (!end.equals(interval.end)) return false;

        return true;
    }

    public static Interval seconds(long beginSeconds, long endSeconds) {
        return new Interval(Instant.ofEpochSecond(beginSeconds), Instant.ofEpochSecond(endSeconds));
    }

    public static Interval millis(long beginMillis, long endMillis) {
        return new Interval(Instant.ofEpochMilli(beginMillis), Instant.ofEpochMilli(endMillis));
    }

    @Override
    public int hashCode() {
        int result = begin.hashCode();
        result = 31 * result + end.hashCode();
        return result;
    }

    public long countPoints(Duration step) {
        long stepWidth = step.getSeconds();
        return (end.getEpochSecond() - 1) / stepWidth - (begin.getEpochSecond() - 1) / stepWidth;
    }

    public long getPointNumber(Instant point, Duration step) {
        long stepWidth = step.getSeconds();
        return point.getEpochSecond() / stepWidth - begin.getEpochSecond() / stepWidth;
    }

    /**
     * @return interval that include both interval
     */
    public static Interval convexHull(Interval first, Interval second) {
        Instant extendBegin = first.begin.isBefore(second.begin) ? first.begin : second.begin;
        Instant extendEnd = first.end.isAfter(second.end) ? first.end : second.end;

        return new Interval(extendBegin, extendEnd);
    }

    public static Interval truncate(Interval interval, long gridMillis) {
        Instant begin = InstantUtils.truncate(interval.getBegin(), gridMillis);
        Instant end  = InstantUtils.truncate(interval.getEnd(), gridMillis);
        return new Interval(begin, end);
    }

    public static Interval extendToGrid(Interval interval, long gridMillis) {
        Instant begin = InstantUtils.truncate(interval.getBegin(), gridMillis);
        Instant end  = InstantUtils.ceil(interval.getEnd(), gridMillis);
        return new Interval(begin, end);
    }

}
