package ru.yandex.solomon.util;

import org.junit.Assert;
import org.junit.Test;

/**
 * @author Oleg Baryshnikov
 */
public class ServicePathEncoderTest {

    @Test
    public void root() throws Exception {
        Assert.assertEquals("/", ServicePathEncoder.encode("/"));
    }

    @Test
    public void path() throws Exception {
        Assert.assertEquals("/pbbin", ServicePathEncoder.encode("/pbbin"));
    }

    @Test
    public void pathWithFinishSlash() throws Exception {
        Assert.assertEquals("/pbbin/", ServicePathEncoder.encode("/pbbin/"));
    }

    @Test
    public void pathWithQuery() throws Exception {
        String url = "/pbbin/?format=SPACK";
        Assert.assertEquals(url, ServicePathEncoder.encode(url));
    }

    @Test
    public void pathWithEqual() throws Exception {
        String encodedUrl = "/counters/counters=interconnect/spack";
        Assert.assertEquals(encodedUrl, ServicePathEncoder.encode(encodedUrl));
    }

    @Test
    public void queryOnly() throws Exception {
        Assert.assertEquals("?format=SPACK", ServicePathEncoder.encode("?format=SPACK"));
    }

    @Test
    public void defaultQuery() throws Exception {
        String url = "/metrics?format=SPACK";
        Assert.assertEquals(url, ServicePathEncoder.encode(url));
    }

    @Test
    public void queryWithSeveralParams() throws Exception {
        String url = "/metrics?format=SPACK&version=2";
        Assert.assertEquals(url, ServicePathEncoder.encode(url));
    }

    @Test
    public void queryWithKeyWithoutValue() throws Exception {
        String url = "/metrics?format";
        Assert.assertEquals(url, ServicePathEncoder.encode(url));
    }

    @Test(expected = IllegalArgumentException.class)
    public void queryWithEscapedPercent() throws Exception {
        String url = "/?regex=100%";
        String encodedUrl = "/?regex=100%25";
        Assert.assertEquals(encodedUrl, ServicePathEncoder.encode(url));
    }

    @Test
    public void urlEncodedQuery() throws Exception {
        String url = "/?porto.containerName=.*%28yt_%5Cw%2Bnodes%29.*%2Fiss_hook_start";
        Assert.assertEquals(url, ServicePathEncoder.encode(url));
    }

    @Test
    public void query() throws Exception {
        String url = "/api/v1/metrics.json?local=1&grep=^((?!Task:).)*$&solomon=1";
        String encodedUrl = "/api/v1/metrics.json?local=1&grep=%5E%28%28%3F%21Task%3A%29.%29*%24&solomon=1";
        Assert.assertEquals(encodedUrl, ServicePathEncoder.encode(url));
    }

    @Test
    public void queryWithQuestionMark() throws Exception {
        String url = "/?regex=solomon-man-??";
        String encodedUrl = "/?regex=solomon-man-%3F%3F";
        Assert.assertEquals(encodedUrl, ServicePathEncoder.encode(url));
    }

    @Test
    public void queryWithEq() throws Exception {
        String url = "/?regex=counters=interconnect";
        String encodedUrl = "/?regex=counters%3Dinterconnect";
        Assert.assertEquals(encodedUrl, ServicePathEncoder.encode(url));
    }

    // Known collision in new format
    // We always encode '+' as '%20', because we want to support query: "?regex=solomon-.+"
    // It may break service creation with default URL encoders: '?name=my+own+name'
    // You can replace "+" to "%20" in encoded "path" to fix this problem
    @Test
    public void queryWithPlus() throws Exception {
        String url = "/?regex=solomon+man";
        String encodedUrl = "/?regex=solomon%2Bman";
        Assert.assertEquals(encodedUrl, ServicePathEncoder.encode(url));
    }
}
