package ru.yandex.solomon.util;

import org.apache.commons.lang3.RandomStringUtils;
import org.junit.Test;

import static org.junit.Assert.assertEquals;
import static ru.yandex.solomon.util.StringUtils.replaceChars;

/**
 * @author Stanislav Kashirin
 */
public class StringUtilsTest {

    @Test
    public void empty() {
        assertEquals("", replaceChars("", ch -> true, '_'));
        assertEquals("", replaceChars("", ch -> false, '$'));
    }

    @Test
    public void single() {
        assertEquals("_", replaceChars("a", ch -> ch == 'a', '_'));
        assertEquals("a", replaceChars("a", ch -> ch == 'b', '_'));
        assertEquals("$", replaceChars("s", ch -> ch == 's', '$'));
    }

    @Test
    public void atStart() {
        assertEquals("_b", replaceChars("ab", ch -> ch == 'a', '_'));
        assertEquals("_bc", replaceChars("abc", ch -> ch == 'a', '_'));
    }

    @Test
    public void atSecond() {
        assertEquals("a_c1", replaceChars("abc1", ch -> ch == 'b', '_'));
        assertEquals("h_llo", replaceChars("hello", ch -> ch == 'e', '_'));
    }

    @Test
    public void atEnd() {
        assertEquals("a_", replaceChars("ab", ch -> ch == 'b', '_'));
        assertEquals("ab_", replaceChars("abc", ch -> ch == 'c', '_'));
        assertEquals("ab123_", replaceChars("ab123c", ch -> ch == 'c', '_'));
        assertEquals("ab_123_", replaceChars("abc123c", ch -> ch == 'c', '_'));
    }

    @Test
    public void atPreLast() {
        assertEquals("ab!1", replaceChars("abc1", ch -> ch == 'c', '!'));
        assertEquals("swo!d", replaceChars("sword", ch -> ch == 'r', '!'));
    }

    @Test
    public void neighbors() {
        assertEquals("he__o", replaceChars("hello", ch -> ch == 'l', '_'));
    }

    @Test
    public void alphanumeric() {
        var s = RandomStringUtils.randomAlphanumeric(100);

        var jdk = s.replaceAll("[^\\w\\-_+./]", "_");
        var bicycle = replaceChars(s, StringUtilsTest::badChars, '_');

        assertEquals(jdk, bicycle);
    }

    @Test
    public void ascii() {
        var s = RandomStringUtils.randomAscii(100);

        var jdk = s.replaceAll("[^\\w\\-_+./]", "_");
        var bicycle = replaceChars(s, StringUtilsTest::badChars, '_');

        assertEquals(jdk, bicycle);
    }

    @SuppressWarnings({"OverlyComplexBooleanExpression", "OverlyComplexMethod"})
    private static boolean badChars(char ch) {
        return !(ch >= 'a' && ch <= 'z'
            || ch >= 'A' && ch <= 'Z'
            || ch >= '0' && ch <= '9'
            || ch == '_' || ch == '+' || ch == '-' || ch == '.' || ch == '/');
    }
}
