package ru.yandex.solomon.util.actors;

import java.util.Arrays;
import java.util.Collection;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Supplier;

import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;
import org.junit.runners.Parameterized.Parameter;
import org.junit.runners.Parameterized.Parameters;

import ru.yandex.misc.concurrent.CompletableFutures;


/**
 * @author Sergey Polovko
 */
@RunWith(Parameterized.class)
public class AsyncActorRunnerTest {

    @Parameter
    public int maxInFlight;

    @Parameters(name = "maxInFlight={0}")
    public static Collection<Integer> parameters() {
        return Arrays.asList(1, 2, 4, 8);
    }

    private ExecutorService actorExecutor;
    private ScheduledExecutorService schedulerExecutor;

    @Before
    public void beforeTest() {
        actorExecutor = Executors.newFixedThreadPool(2);
        schedulerExecutor = Executors.newScheduledThreadPool(2);
    }

    @After
    public void afterTest() throws InterruptedException {
        actorExecutor.shutdown();
        schedulerExecutor.shutdown();

        Assert.assertTrue(actorExecutor.awaitTermination(5, TimeUnit.MINUTES));
        Assert.assertTrue(schedulerExecutor.awaitTermination(5, TimeUnit.MINUTES));
    }

    @Test
    public void alreadyCompleted() {
        CountableActor a = new CountableActor() {
            @Override
            CompletableFuture<?> doRun() {
                return CompletableFuture.completedFuture(DONE_MARKER);
            }
        };

        AsyncActorRunner r = new AsyncActorRunner(a, actorExecutor, maxInFlight);
        r.start().join();

        Assert.assertEquals(1, a.getStarted());
        Assert.assertEquals(1, a.getFinished());
    }

    @Test
    public void alreadyFailed() {
        CountableActor a = new CountableActor() {
            @Override
            CompletableFuture<?> doRun() {
                return CompletableFuture.failedFuture(new RuntimeException("some error in async operation"));
            }
        };

        AsyncActorRunner r = new AsyncActorRunner(a, actorExecutor, maxInFlight);
        try {
            r.start().join();
        } catch (Throwable t) {
            t = CompletableFutures.unwrapCompletionException(t);
            Assert.assertEquals("some error in async operation", t.getMessage());

            Assert.assertEquals(1, a.getStarted());
            Assert.assertEquals(1, a.getFinished());
            return;
        }

        Assert.fail("exception is not rethrown");
    }

    @Test
    public void completedAfterSomeTime() {
        CountableActor a = new CountableActor() {
            @Override
            CompletableFuture<?> doRun() {
                return supplyAfter(200, TimeUnit.MILLISECONDS, () -> DONE_MARKER);
            }
        };

        AsyncActorRunner r = new AsyncActorRunner(a, actorExecutor, maxInFlight);
        r.start().join();

        Assert.assertEquals(maxInFlight, a.getStarted());
        Assert.assertEquals(maxInFlight, a.getFinished());
    }

    @Test
    public void failedAfterSomeTime() {
        CountableActor a = new CountableActor() {
            @Override
            CompletableFuture<?> doRun() {
                return supplyAfter(200, TimeUnit.MILLISECONDS, () -> {
                    throw new RuntimeException("some error in async operation");
                });
            }
        };

        AsyncActorRunner r = new AsyncActorRunner(a, actorExecutor, maxInFlight);
        try {
            r.start().join();
        } catch (Throwable t) {
            t = CompletableFutures.unwrapCompletionException(t);
            Assert.assertEquals("some error in async operation", t.getMessage());

            Assert.assertEquals(maxInFlight, a.getStarted());
            Assert.assertTrue(maxInFlight >= a.getFinished()); // because on error we do not wait all async operations

            return;
        }

        Assert.fail("exception is not rethrown");
    }

    @Test
    public void completedFutureEachTime() {
        AtomicInteger count = new AtomicInteger();
        AsyncActorBody body = () -> {
            if (count.incrementAndGet() > 3) {
                return CompletableFuture.completedFuture(AsyncActorBody.DONE_MARKER);
            }

            return CompletableFuture.completedFuture(null);
        };

        new AsyncActorRunner(body, actorExecutor, maxInFlight).start().join();
    }

    private <V> CompletableFuture<V> supplyAfter(long delay, TimeUnit unit, Supplier<V> supplier) {
        CompletableFuture<V> future = new CompletableFuture<>();
        schedulerExecutor.schedule(() -> {
            try {
                V result = supplier.get();
                future.complete(result);
            } catch (Throwable t) {
                future.completeExceptionally(t);
            }
        }, delay, unit);
        return future;
    }

    /**
     * COUNTABLE ACTOR
     */
    private abstract class CountableActor implements AsyncActorBody {

        private final AtomicInteger started = new AtomicInteger();
        private final AtomicInteger finished = new AtomicInteger();

        @Override
        public final CompletableFuture<?> run() {
            started.incrementAndGet();
            return doRun()
                .whenComplete((r, t) -> {
                    finished.incrementAndGet();
                });
        }

        final int getStarted() {
            return started.get();
        }

        final int getFinished() {
            return finished.get();
        }

        abstract CompletableFuture<?> doRun();
    }
}
