package ru.yandex.solomon.util.actors;

import java.time.Duration;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.Semaphore;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.IntFunction;

import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.Timeout;

import ru.yandex.solomon.ut.ManualClock;
import ru.yandex.solomon.ut.ManualScheduledExecutorService;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static org.hamcrest.Matchers.greaterThanOrEqualTo;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertThat;

/**
 * @author Vladimir Gordiychuk
 */
public class PingActorRunnerTest {

    @Rule
    public Timeout timeout = Timeout.builder()
            .withTimeout(15, TimeUnit.SECONDS)
            .withLookingForStuckThread(true)
            .build();

    private ManualClock clock;
    private ManualScheduledExecutorService timer;

    @Before
    public void setUp() throws Exception {
        clock = new ManualClock();
        timer = new ManualScheduledExecutorService(1, clock);
    }

    @Test
    public void pingByInterval() {
        AtomicInteger ping = new AtomicInteger();
        Semaphore sem1 = new Semaphore(0);
        Semaphore sem2 = new Semaphore(0);
        IntFunction<CompletableFuture<?>> supplier = (ignore) -> {
            wrapInterruptable(sem1::acquire);
            ping.incrementAndGet();
            sem2.release();
            return completedFuture(null);
        };

        try (var actor = PingActorRunner.newBuilder()
                .pingInterval(Duration.ofMillis(5))
                .onPing(supplier)
                .timer(timer)
                .executor(ForkJoinPool.commonPool())
                .build())
        {
            actor.schedule();
            sem1.release();
            for (int index = 1; index <= 3; index++) {
                wrapInterruptable(sem2::acquire);
                assertEquals(ping.get(), index);
                sem1.release();
            }
        }
    }

    @Test
    public void continuePingAfterError() throws InterruptedException {
        AtomicInteger ping = new AtomicInteger();
        CountDownLatch sync = new CountDownLatch(5);
        IntFunction<CompletableFuture<?>> supplier = (ignore) -> {
            return CompletableFuture.supplyAsync(() -> {
                if (ThreadLocalRandom.current().nextBoolean()) {
                    throw new IllegalStateException("test");
                }

                ping.incrementAndGet();
                sync.countDown();
                return null;
            });
        };

        try (var actor = PingActorRunner.newBuilder()
                .pingInterval(Duration.ofMillis(5))
                .onPing(supplier)
                .timer(timer)
                .backoffDelay(Duration.ofMillis(1))
                .backoffMaxDelay(Duration.ofMillis(3))
                .executor(ForkJoinPool.commonPool())
                .build())
        {
            actor.schedule();
            sync.await();
            assertThat(ping.get(), greaterThanOrEqualTo(5));
        }
    }

    @Test
    public void retryPing() throws InterruptedException {
        AtomicInteger ping = new AtomicInteger();
        CountDownLatch sync = new CountDownLatch(1);
        IntFunction<CompletableFuture<?>> supplier = (ignore) -> {
            return CompletableFuture.supplyAsync(() -> {
                if (ping.incrementAndGet() < 5) {
                    throw new IllegalStateException("test");
                }

                sync.countDown();
                return null;
            });
        };

        try (var actor = PingActorRunner.newBuilder()
                .pingInterval(Duration.ofMinutes(5))
                .onPing(supplier)
                .timer(timer)
                .backoffDelay(Duration.ofMillis(1))
                .backoffMaxDelay(Duration.ofMillis(3))
                .executor(ForkJoinPool.commonPool())
                .build())
        {
            actor.forcePing();
            sync.await();
            assertEquals(5, ping.get());
        }
    }

    @Test
    public void retryPingWithSupplierException() throws InterruptedException {
        AtomicInteger ping = new AtomicInteger();
        CountDownLatch sync = new CountDownLatch(1);
        IntFunction<CompletableFuture<?>> supplier = (ignore) -> {
            if (ping.incrementAndGet() < 5) {
                throw new IllegalStateException("test");
            }

            return CompletableFuture.supplyAsync(() -> {
                sync.countDown();
                return null;
            });
        };

        try (var actor = PingActorRunner.newBuilder()
                .pingInterval(Duration.ofMinutes(5))
                .onPing(supplier)
                .timer(timer)
                .backoffDelay(Duration.ofMillis(1))
                .backoffMaxDelay(Duration.ofMillis(3))
                .executor(ForkJoinPool.commonPool())
                .build())
        {
            actor.forcePing();
            sync.await();
            assertEquals(5, ping.get());
        }
    }

    private interface Interruptable {
        public void run() throws InterruptedException;
    }

    private void wrapInterruptable(Interruptable r) {
        try {
            r.run();
        } catch (InterruptedException e) {
            throw new RuntimeException(e);
        }
    }
}
