package ru.yandex.solomon.util.actors;

import java.util.Arrays;
import java.util.Queue;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ForkJoinPool;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.LongAdder;
import java.util.function.Consumer;
import java.util.stream.IntStream;

import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.misc.actor.ActorRunnerImpl;

/**
 * @author Egor Litvinenko
 * */
public class SimpleQueueActorTest {

    private ExecutorService oneExecutor;

    @Before
    public void beforeTest() {
        oneExecutor = new ForkJoinPool(4);
    }

    @After
    public void afterTest() throws InterruptedException {
        oneExecutor.shutdown();
        Assert.assertTrue(oneExecutor.awaitTermination(5, TimeUnit.MINUTES));
    }

    @Test
    public void SingleThreadBaselineActor_FullOrder() throws Exception {
        RaceIntegers race = new RaceIntegers();
        final int tasks = 10000;
        final int tickLimit = 10;
        final int[] numbers = new int[tasks];
        SingleThreadBaselineActor actor = new SingleThreadBaselineActor(oneExecutor, tickLimit, e -> e.printStackTrace());
        for (int i = 0; i < tasks; i++) {
            final int index = i;
            actor.enqueue(() -> {
                race.count = race.count + 1;
                race.sum = race.sum + race.count;
                numbers[index] = race.count;
            });
        }
        Thread.sleep(500);
        Assert.assertEquals(tasks, race.count);
        Assert.assertEquals(tasks * (1 + tasks) / 2, race.sum);
        int[] expected = IntStream.rangeClosed(1, tasks).toArray();
        Assert.assertTrue(Arrays.equals(expected, numbers));
    }

    @Test
    public void SingleThreadBaselineActor_RandomOrder() throws Exception {
        RaceIntegers race = new RaceIntegers();
        final int tasks = 10000;
        final int tickLimit = 10;
        final int[] numbers = new int[tasks];
        SingleThreadBaselineActor actor = new SingleThreadBaselineActor(oneExecutor, tickLimit, e -> e.printStackTrace());
        for (int i = 0; i < tasks; i++) {
            final int index = i;
            CompletableFuture.runAsync(() -> actor.enqueue(() -> {
                race.count = race.count + 1;
                race.sum = race.sum + race.count;
                numbers[index] = race.count;
            }), oneExecutor);
        }
        Thread.sleep(500);
        Assert.assertEquals(tasks, race.count);
        Assert.assertEquals(tasks * (1 + tasks) / 2, race.sum);
    }

    @Test
    public void SimpleQueueActor_RandomOrder() throws Exception {
        RaceIntegers race = new RaceIntegers();
        final int tasks = 10000;
        final int[] numbers = new int[tasks];
        var actor = new SimpleQueueActor<Integer>(oneExecutor,
                index -> {
                    race.count = race.count + 1;
                    race.sum = race.sum + race.count;
                    numbers[index] = race.count;
                },
                e -> e.printStackTrace(),
                10);
        for (int i = 0; i < tasks; i++) {
            final int index = i;
            CompletableFuture.runAsync(() -> actor.enqueue(Integer.valueOf(index)), oneExecutor);
        }
        Thread.sleep(500);
        Assert.assertEquals(tasks, race.count);
        Assert.assertEquals(tasks * (1 + tasks) / 2, race.sum);
//        Assert.assertTrue(Arrays.equals(IntStream.rangeClosed(1, tasks).toArray(), numbers));
    }

    @Test
    public void SimpleQueueActor_FullOrder() throws Exception {
        RaceIntegers race = new RaceIntegers();
        final int tasks = 10000;
        final int[] numbers = new int[tasks];
        var actor = new SimpleQueueActor<Integer>(oneExecutor,
                index -> {
                    race.count = race.count + 1;
                    race.sum = race.sum + race.count;
                    numbers[index] = race.count;
                },
                e -> e.printStackTrace(),
                10);
        for (int i = 0; i < tasks; i++) {
            final int index = i;
            actor.enqueue(index);
        }
        Thread.sleep(500);
        Assert.assertEquals(tasks, race.count);
        Assert.assertEquals(tasks * (1 + tasks) / 2, race.sum);
        Assert.assertTrue(Arrays.equals(IntStream.rangeClosed(1, tasks).toArray(), numbers));
    }

    private static final class RaceIntegers {
        int count;
        int sum;
    }

    private static class SingleThreadBaselineActor implements Runnable {

        private final Executor executor;
        private final Queue<Runnable> tasks;
        private final LongAdder size;
        private final AtomicBoolean scheduled;
        private final Consumer<Throwable> exceptionHandler;
        private final int executionTickLimit;

        public SingleThreadBaselineActor(Executor executor, int executionTickLimit, Consumer<Throwable> exceptionHandler) {
            this.executor = executor;
            this.executionTickLimit = executionTickLimit;
            this.size = new LongAdder();
            this.tasks = new ConcurrentLinkedQueue<>();
            this.scheduled = new AtomicBoolean(false);
            this.exceptionHandler = exceptionHandler;
        }

        public void enqueue(Runnable task) {
            tasks.add(task);
            size.increment();
            if (scheduled.compareAndSet(false, true)) {
                ActorRunnerImpl.schedule(executor, this);
            }
        }

        public int size() {
            return size.intValue();
        }

        @Override
        public void run() {
            Runnable task = tasks.poll();
            if (task == null) {
                this.scheduled.setRelease(false);
                return;
            }
            try {
                final int tickLimit = executionTickLimit;
                int i = 0;
                do {
                    size.decrement();
                    task.run();
                } while (++i < tickLimit && (task = tasks.poll()) != null);
            } catch (Throwable throwable) {
                exceptionHandler.accept(throwable);
            } finally {
                if (task == null) {
                    this.scheduled.setRelease(false);
                } else {
                    ActorRunnerImpl.schedule(executor, this);
                }
            }
        }

//    flush example
// this flush gives race on executionTickLimit, but...
// If we have broken value due to drain, in the worse case we just run the actor twice
// Sequence of these runnings are guaranteed by cycle on this.schedule

//    /**
//     * If there is execution wait it until timeout.
//     * If timeout is reached - return.
//     * Otherwise, run processing for all queue in the current thread.
//     * */
//    public void flush(long timeout, TimeUnit timeUnit) {
//        final long time = System.nanoTime();
//        final long due = time + timeUnit.toNanos(timeout);
//        while (this.scheduled.get() && System.nanoTime() < due) {
//            LockSupport.parkNanos(TimeUnit.MILLISECONDS.toNanos(100));
//        }
//        if (!this.scheduled.get() && System.nanoTime() < due) {
//            this.executionTickLimit = Integer.MAX_VALUE;
//            run();
//        }
//    }
    }


}
