package ru.yandex.solomon.util.client;

import java.util.List;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;

import io.grpc.Status;
import org.junit.Test;

import static java.util.concurrent.CompletableFuture.completedFuture;
import static java.util.concurrent.CompletableFuture.delayedExecutor;
import static java.util.concurrent.CompletableFuture.failedFuture;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static ru.yandex.solomon.util.client.ClientFutures.allOrAnyOkOf;

/**
 * @author Stanislav Kashirin
 */
public class ClientFuturesTest {

    private final Response ok = new Response(true);
    private final Response fail = new Response(false);

    @Test
    public void allOfOrAny1() {
        // OK
        {
            var responsesFuture = allOrAnyOkOf(List.of(completedFuture(ok)), () -> 10_000);
            // immediately completed future
            assertTrue(responsesFuture.isDone());
            assertEquals(List.of(ok), responsesFuture.join());
        }

        // FAIL
        {
            var responsesFuture = allOrAnyOkOf(List.of(completedFuture(fail)), () -> 10_000);
            // immediately completed future
            assertTrue(responsesFuture.isDone());
            assertEquals(List.of(fail), responsesFuture.join());
        }
    }

    @Test
    public void allOfOrAny2() {
        long normal = 200;
        long tooLong = 600;
        long max = 2000;

        // OK, FAIL
        {
            var responsesFuture = allOrAnyOkOf(
                List.of(
                    completedFuture(ok),
                    completedFuture(fail)),
                () -> max);
            // immediately completed future
            assertTrue(responsesFuture.isDone());
            assertEquals(List.of(ok, fail), responsesFuture.join());
        }

        // OK, FAIL (delayed < maxTime)
        {
            var responses = allOrAnyOkOf(
                List.of(
                    delayedResponse(fail, normal),
                    completedFuture(ok)),
                () -> tooLong).join();
            assertEquals(List.of(fail, ok), responses);
        }


        // OK, FAIL (delayed > maxTime)
        {
            var responses = allOrAnyOkOf(
                List.of(
                    delayedResponse(fail, tooLong),
                    completedFuture(ok)),
                () -> normal).join();
            assertEquals(List.of(ok), responses);
        }

        // OK (delayed > maxTime), FAIL
        {
            var responses = allOrAnyOkOf(
                List.of(
                    delayedResponse(ok, tooLong),
                    completedFuture(fail)),
                () -> normal).join();
            // anyway we await OK response
            assertEquals(List.of(ok, fail), responses);
        }
    }

    @Test
    public void allOfOrAny3() {
        long fast = 100;
        long normal = 200;
        long tooLong = 1000;

        {
            var responses = allOrAnyOkOf(
                List.of(
                    completedFuture(ok.from(1)),
                    delayedResponse(ok.from(2), tooLong),
                    delayedResponse(fail.from(3), fast)),
                () -> normal).join();

            assertEquals(List.of(ok.from(1), fail.from(3)), responses);
        }

        {
            var responses = allOrAnyOkOf(
                List.of(
                    completedFuture(ok.from(1)),
                    delayedResponse(fail.from(2), tooLong),
                    delayedResponse(fail.from(3), fast)),
                () -> normal).join();

            assertEquals(List.of(ok.from(1), fail.from(3)), responses);
        }

        {
            var responses = allOrAnyOkOf(
                List.of(
                    delayedResponse(ok.from(1), fast),
                    delayedResponse(fail.from(2), fast),
                    delayedResponse(fail.from(3), tooLong)),
                () -> normal).join();

            assertEquals(List.of(ok.from(1), fail.from(2)), responses);
        }
    }

    @Test
    public void withException() {
        long fast = 100;
        long normal = 200;
        long tooLong = 1000;

        {
            var responses = allOrAnyOkOf(
                List.of(
                    failedFuture(new RuntimeException("1")),
                    delayedResponse("2", tooLong),
                    delayedResponse("3", fast)),
                () -> normal).join();

            assertEquals(List.of("3"), responses);
        }

        {
            var responses = allOrAnyOkOf(
                List.of(
                    delayedResponse("1", fast),
                    delayedResponse("2", normal),
                    failedFuture(new RuntimeException("3"))),
                () -> tooLong).join();

            assertEquals(List.of("1", "2"), responses);
        }

        {
            var responses = allOrAnyOkOf(
                List.of(
                    failedFuture(new RuntimeException("1")),
                    delayedResponse(ok.from(2), normal),
                    delayedResponse(ok.from(3), fast)),
                () -> tooLong).join();

            assertEquals(List.of(ok.from(2), ok.from(3)), responses);
        }

        {
            var responses = allOrAnyOkOf(
                List.of(
                    failedFuture(new RuntimeException("1")),
                    delayedResponse(fail.from(2), normal),
                    delayedResponse(ok.from(3), fast)),
                () -> tooLong).join();

            assertEquals(List.of(fail.from(2), ok.from(3)), responses);
        }

        {
            var responses = allOrAnyOkOf(
                List.of(
                    failedFuture(new RuntimeException("1")),
                    delayedResponse(ok.from(2), tooLong),
                    delayedResponse(ok.from(3), fast)),
                () -> normal).join();

            assertEquals(List.of(ok.from(3)), responses);
        }

        {
            var f = allOrAnyOkOf(
                List.of(
                    failedFuture(new RuntimeException("1")),
                    delayedResponse(fail.from(2), fast),
                    delayedResponse(fail.from(3), fast)),
                () -> normal);

           assertFalse(asStatus(f).isOk());
        }

        {
            var f = allOrAnyOkOf(
                List.of(
                    failedFuture(new RuntimeException("1")),
                    delayedException(new RuntimeException("2"), fast),
                    delayedException(new RuntimeException("3"), fast)),
                () -> normal);

            assertFalse(asStatus(f).isOk());
        }
    }

    private static <T> CompletableFuture<T> delayedResponse(T r, long delayMillis) {
        var f = new CompletableFuture<T>();
        delayedExecutor(delayMillis, TimeUnit.MILLISECONDS)
            .execute(() -> f.complete(r));
        return f;
    }

    private static <T extends Throwable> CompletableFuture<T> delayedException(T t, long delayMillis) {
        var f = new CompletableFuture<T>();
        delayedExecutor(delayMillis, TimeUnit.MILLISECONDS)
            .execute(() -> f.completeExceptionally(t));
        return f;
    }

    private static Status asStatus(CompletableFuture<?> f) {
        return f.thenApply(i -> Status.OK).exceptionally(Status::fromThrowable).join();
    }

    private static final class Response implements StatusAware {
        private final boolean ok;
        private final int id;

        private Response(boolean ok) {
            this(ok, 0);
        }

        private Response(boolean ok, int id) {
            this.ok = ok;
            this.id = id;
        }

        private Response from(int id) {
            return new Response(ok, id);
        }

        @Override
        public boolean isOk() {
            return ok;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || getClass() != o.getClass()) {
                return false;
            }
            Response response = (Response) o;
            return ok == response.ok &&
                id == response.id;
        }

        @Override
        public int hashCode() {
            return Objects.hash(ok, id);
        }

        @Override
        public String toString() {
            return "Response{ok=" + ok + ", id=" + id + '}';
        }
    }

}
