package ru.yandex.solomon.util.collection.unpacked.arrayList;

import java.util.ArrayList;
import java.util.List;
import java.util.Random;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import org.junit.Assert;
import org.junit.Ignore;
import org.junit.Test;

import ru.yandex.misc.random.Random2;
import ru.yandex.misc.test.ForTestSynchronizationUtility;

/**
 * @author Stepan Koltsov
 */
public class UnpackedArrayListTest {

    private static class MyItem {
        private final int a;
        private final String b;

        public MyItem(int a, String b) {
            this.a = a;
            this.b = b;
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) return true;
            if (o == null || getClass() != o.getClass()) return false;

            MyItem myItem = (MyItem) o;

            if (a != myItem.a) return false;
            return !(b != null ? !b.equals(myItem.b) : myItem.b != null);

        }

        @Override
        public int hashCode() {
            int result = a;
            result = 31 * result + (b != null ? b.hashCode() : 0);
            return result;
        }

        @Override
        public String toString() {
            return "MyItem{" +
                "a=" + a +
                ", b='" + b + '\'' +
                '}';
        }
    }

    @Test
    public void simple() {
        UnpackedArrayList<MyItem> arrayList = new UnpackedArrayList.Factory<>(MyItem.class).newInstance();

        arrayList.add(new MyItem(10, "a"));
        arrayList.add(new MyItem(20, "b"));

        Assert.assertEquals(2, arrayList.size());
        Assert.assertEquals(new MyItem(10, "a"), arrayList.get(0));
        Assert.assertEquals(new MyItem(20, "b"), arrayList.get(1));
    }

    @Test
    public void withCapacity() {
        UnpackedArrayList<MyItem> arrayList = new UnpackedArrayList.Factory<>(MyItem.class).newWithCapacity(10);

        arrayList.add(new MyItem(10, "a"));
        arrayList.add(new MyItem(20, "b"));

        Assert.assertEquals(2, arrayList.size());
        Assert.assertEquals(new MyItem(10, "a"), arrayList.get(0));
        Assert.assertEquals(new MyItem(20, "b"), arrayList.get(1));
    }

    @Test
    public void addAll() {
        UnpackedArrayList<MyItem> arrayList = new UnpackedArrayList.Factory<>(MyItem.class).newWithCapacity(10);

        arrayList.addAll(new MyItem(10, "a"), new MyItem(20, "b"));

        Assert.assertEquals(2, arrayList.size());
        Assert.assertEquals(new MyItem(10, "a"), arrayList.get(0));
        Assert.assertEquals(new MyItem(20, "b"), arrayList.get(1));
    }

    @Test
    public void trimToSize() {
        UnpackedArrayList<MyItem> arrayList = new UnpackedArrayList.Factory<>(MyItem.class).newInstance();
        arrayList.reserveAdditional(10);
        Assert.assertEquals(10, arrayList.capacity());

        arrayList.add(new MyItem(1, "a"));
        arrayList.add(new MyItem(2, "b"));
        arrayList.trimToSize();

        Assert.assertEquals(2, arrayList.capacity());
        Assert.assertEquals(2, arrayList.size());
        Assert.assertEquals(new MyItem(1, "a"), arrayList.get(0));
        Assert.assertEquals(new MyItem(2, "b"), arrayList.get(1));

        arrayList.add(new MyItem(3, "c"));

        Assert.assertEquals(3, arrayList.size());
        Assert.assertEquals(new MyItem(1, "a"), arrayList.get(0));
        Assert.assertEquals(new MyItem(2, "b"), arrayList.get(1));
        Assert.assertEquals(new MyItem(3, "c"), arrayList.get(2));
    }

    @Test
    public void streamIsParallel() {
        UnpackedArrayList<MyItem> arrayList = new UnpackedArrayList.Factory<>(MyItem.class).newInstance();

        arrayList.add(new MyItem(1, "a"));
        arrayList.add(new MyItem(2, "b"));

        ForTestSynchronizationUtility u = new ForTestSynchronizationUtility();

        arrayList.parallelStream().forEach(item -> {
            if (item.a == 1) {
                u.take(0);
            }
            if (item.a == 2) {
                u.take(1);
            }
            if (item.a == 1) {
                u.take(2);
            }
            if (item.a == 2) {
                u.take(3);
            }
        });

        u.takeNow(4);
    }

    @Test
    public void swap() {
        UnpackedArrayList<MyItem> arrayList = new UnpackedArrayList.Factory<>(MyItem.class).newInstance();

        arrayList.add(new MyItem(1, "x"));
        arrayList.add(new MyItem(2, "y"));
        arrayList.add(new MyItem(3, "z"));

        arrayList.swapForTest(1, 2);

        List<MyItem> expected = new ArrayList<>();
        expected.add(new MyItem(1, "x"));
        expected.add(new MyItem(3, "z"));
        expected.add(new MyItem(2, "y"));

        Assert.assertEquals(expected, arrayList);
    }

    private static int compare(MyItem x, MyItem y) {
        int c = Integer.compare(x.a, y.a);
        if (c != 0) {
            return c;
        }

        return x.b.compareTo(y.b);
    }

    private void sortTestImpl(List<MyItem> items) {
        UnpackedArrayList<MyItem> unpacked = new UnpackedArrayList.Factory<>(MyItem.class).fromList(items);
        ArrayList<MyItem> arrayList = new ArrayList<>(items);

        arrayList.sort(UnpackedArrayListTest::compare);
        unpacked.sort(UnpackedArrayListTest::compare);

        Assert.assertEquals((List<MyItem>) arrayList, unpacked);
    }

    @Test
    public void sort() {
        sortTestImpl(List.of());
        sortTestImpl(List.of(new MyItem(1, "x")));
        sortTestImpl(List.of(new MyItem(1, "x"), new MyItem(2, "y")));
        sortTestImpl(List.of(new MyItem(2, "y"), new MyItem(1, "x")));
        sortTestImpl(List.of(new MyItem(1, "x"), new MyItem(2, "y"), new MyItem(3, "z")));
        sortTestImpl(List.of(new MyItem(3, "z"), new MyItem(2, "y"), new MyItem(1, "x")));
    }

    private MyItem randomItem(Random random) {
        return new MyItem(random.nextInt(1000), new Random2(random).nextAlnum(random.nextInt(5)));
    }

    private void sortRandomIter(Random random) {
        int length = random.nextInt(1000);
        List<MyItem> list = IntStream.range(0, length)
            .mapToObj(i -> randomItem(random))
            .collect(Collectors.toList());

        sortTestImpl(list);
    }

    @Test
    public void sortRandom() {
        Random random = new Random();
        for (int i = 0; i < 100; ++i) {
            sortRandomIter(random);
        }
    }

    private class MyItemExt extends MyItem {
        private MyItemExt(int a, String b) {
            super(a, b);
        }
    }

    @Test
    @Ignore
    public void extendingClass() {
        UnpackedArrayList<MyItemExt> arrayList = new UnpackedArrayList.Factory<>(MyItemExt.class).newInstance();

        arrayList.add(new MyItemExt(10, "a"));
        arrayList.add(new MyItemExt(20, "b"));

        Assert.assertEquals(2, arrayList.size());
        Assert.assertEquals(new MyItemExt(10, "a"), arrayList.get(0));
        Assert.assertEquals(new MyItemExt(20, "b"), arrayList.get(1));
    }

}
