package ru.yandex.solomon.util.future;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Supplier;

import javax.annotation.ParametersAreNonnullByDefault;

import org.junit.Assert;
import org.junit.Test;

import ru.yandex.devtools.test.annotations.YaIgnore;
import ru.yandex.misc.concurrent.CompletableFutures;

/**
 * @author Maksim Leonov (nohttp@)
 */
@ParametersAreNonnullByDefault
public class RetryCompletableFutureTest {
    @Test
    public void simple() {
        AtomicInteger timesRun = new AtomicInteger(0);
        Object expectedValue = new Object();
        Supplier<CompletableFuture<Object>> supplier = () -> {
            timesRun.incrementAndGet();
            return CompletableFuture.completedFuture(expectedValue);
        };
        CompletableFuture<Object> future = RetryCompletableFuture.runWithRetries(supplier, RetryConfig.DEFAULT.withNumRetries(5));
        Object resultValue = CompletableFutures.join(future);
        Assert.assertSame(expectedValue, resultValue);
        Assert.assertEquals(1, timesRun.get());
    }

    @Test
    public void withRetries() {
        AtomicInteger timesRun = new AtomicInteger(0);
        Object expectedValue = new Object();
        Supplier<CompletableFuture<Object>> supplier = () -> {
            if (timesRun.incrementAndGet() < 2) {
                return CompletableFuture.failedFuture(new RuntimeException());
            }
            return CompletableFuture.completedFuture(expectedValue);
        };
        CompletableFuture<Object> future = RetryCompletableFuture.runWithRetries(supplier, RetryConfig.DEFAULT.withNumRetries(3));
        Object resultValue = CompletableFutures.join(future);
        Assert.assertSame(expectedValue, resultValue);
        Assert.assertEquals(2, timesRun.get());
    }

    @Test
    public void manyFailures() {
        AtomicInteger timesRun = new AtomicInteger(0);
        RuntimeException expectedException = new RuntimeException();
        Supplier<CompletableFuture<Object>> supplier = () -> {
            if (timesRun.incrementAndGet() < 4) {
                return CompletableFuture.failedFuture(expectedException);
            }
            return CompletableFuture.completedFuture(new Object());
        };

        CompletableFuture<Object> future = RetryCompletableFuture.runWithRetries(supplier, RetryConfig.DEFAULT.withNumRetries(3));

        try {
            CompletableFutures.join(future);
            Assert.fail("Exception not thrown");
        } catch (RuntimeException e) {
            Assert.assertSame(expectedException, e.getCause().getCause());
        }
        Assert.assertEquals(3, timesRun.get());
    }

    @Test
    @YaIgnore
    public void withIntervals() {
        AtomicInteger timesRun = new AtomicInteger(0);
        int numTries = 3;
        Supplier<CompletableFuture<Object>> supplier = () -> {
            if (timesRun.incrementAndGet() < numTries) {
                return CompletableFuture.failedFuture(new RuntimeException());
            }
            return CompletableFuture.completedFuture(new Object());
        };

        long retryDelayMillis = 1000;

        long startTimeMillis = System.currentTimeMillis();
        CompletableFutures.join(RetryCompletableFuture.runWithRetries(supplier, RetryConfig.DEFAULT.withNumRetries(numTries).withDelay(retryDelayMillis)));
        long timeSpentMillis = System.currentTimeMillis() - startTimeMillis;
        long expectedTimeSpentMillis = (numTries - 1) * retryDelayMillis;

        Assert.assertEquals("Expected to run at least " + expectedTimeSpentMillis + "ms, " + "run for " + timeSpentMillis + "ms",
                timeSpentMillis, expectedTimeSpentMillis, retryDelayMillis);
    }
}
