package ru.yandex.solomon.util.future;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.AtomicInteger;

import javax.annotation.ParametersAreNonnullByDefault;

import org.junit.Assert;
import org.junit.Test;

/**
 * @author Maksim Leonov (nohttp@)
 */
@ParametersAreNonnullByDefault
public class RetryUtilsTest {
    interface TestIface {
        Object method();
        CompletableFuture<Object> futureMethod();
        CompletableFuture<Object> futureMethodWithExceptions();
    }

    class Impl implements TestIface {
        final Object a = new Object();
        final Object b = new Object();
        final Object c = new Object();
        final AtomicInteger methodCallCounter = new AtomicInteger();
        final AtomicInteger futureMethodCallCounter = new AtomicInteger();
        final AtomicInteger futureMethodWithExceptionsCallCounter = new AtomicInteger();

        @Override
        public Object method() {
            methodCallCounter.incrementAndGet();
            return a;
        }

        @Override
        public CompletableFuture<Object> futureMethod() {
            futureMethodCallCounter.incrementAndGet();
            return CompletableFuture.completedFuture(b);
        }

        @Override
        public CompletableFuture<Object> futureMethodWithExceptions() {
            if (futureMethodWithExceptionsCallCounter.incrementAndGet() < 2) {
                return CompletableFuture.failedFuture(new RuntimeException());
            }
            return CompletableFuture.completedFuture(c);
        }
    }

    @Test
    public void testSimpleMethod() {
        Impl impl = new Impl();
        TestIface proxy = RetryUtils.wrapWithRetries(impl, TestIface.class, RetryConfig.DEFAULT);
        Object result = proxy.method();
        Assert.assertSame(impl.a, result);
        Assert.assertEquals(1, impl.methodCallCounter.get());
        Assert.assertEquals(0, impl.futureMethodCallCounter.get());
        Assert.assertEquals(0, impl.futureMethodWithExceptionsCallCounter.get());
    }

    @Test
    public void testFutureMethod() {
        Impl impl = new Impl();
        TestIface proxy = RetryUtils.wrapWithRetries(impl, TestIface.class, RetryConfig.DEFAULT);
        Object result = proxy.futureMethod().join();
        Assert.assertSame(impl.b, result);
        Assert.assertEquals(0, impl.methodCallCounter.get());
        Assert.assertEquals(1, impl.futureMethodCallCounter.get());
        Assert.assertEquals(0, impl.futureMethodWithExceptionsCallCounter.get());
    }

    @Test
    public void testFutureMethodWithExceptions() {
        Impl impl = new Impl();
        TestIface proxy = RetryUtils.wrapWithRetries(impl, TestIface.class, RetryConfig.DEFAULT);
        Object result = proxy.futureMethodWithExceptions().join();
        Assert.assertSame(impl.c, result);
        Assert.assertEquals(0, impl.methodCallCounter.get());
        Assert.assertEquals(0, impl.futureMethodCallCounter.get());
        Assert.assertEquals(2, impl.futureMethodWithExceptionsCallCounter.get());
    }
}
