package ru.yandex.solomon.util.labelStats;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.stream.IntStream;

import org.junit.Test;

import ru.yandex.monlib.metrics.labels.validate.LabelValidationFilter;

import static org.hamcrest.Matchers.allOf;
import static org.hamcrest.Matchers.emptyIterable;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.hasItem;
import static org.hamcrest.Matchers.iterableWithSize;
import static org.hamcrest.Matchers.not;
import static org.junit.Assert.assertThat;

/**
 * @author Vladimir Gordiychuk
 */
public class LabelStatsTest {

    @Test
    public void add() {
        LabelStats stats = LabelStats.create();
        stats.add("one");
        stats.add("two");
        stats.add("tree");
        stats.add("two");

        assertThat("4 metrics have target label, and one of value not unique",
                stats.getCount(), equalTo(4));
        assertThat("Available values contains only unique values",
                stats.getValues(), allOf(hasItem("one"), hasItem("two"), hasItem("tree")));
        assertThat(stats.isTruncated(), equalTo(false));
    }

    @Test
    public void empty() {
        LabelStats stats = LabelStats.create();
        assertThat(stats.getCount(), equalTo(0));
        assertThat(stats.getValues(), emptyIterable());
        assertThat(stats.isTruncated(), equalTo(false));
    }

    @Test
    public void emptyCombine() {
        LabelStats one = LabelStats.create();
        LabelStats two = LabelStats.create();

        LabelStats result = LabelStats.create();
        result.combine(one);
        result.combine(two);

        assertThat(result.getCount(), equalTo(0));
        assertThat(result.getValues(), emptyIterable());
        assertThat(result.isTruncated(), equalTo(false));

    }

    @Test
    public void combine() {
        LabelStats one = LabelStats.create();
        one.add("elapsedTime");
        one.add("requestStarted");

        LabelStats two = LabelStats.create();
        two.add("requestStarted");
        two.add("requestInFlight");
        two.add("requestCompleted");

        LabelStats result = LabelStats.create();
        result.combine(one);
        result.combine(two);

        assertThat(result.getCount(), equalTo(5));
        assertThat(result.getValues(), allOf(
                hasItem("elapsedTime"),
                hasItem("requestStarted"),
                hasItem("requestInFlight"),
                hasItem("requestCompleted")
        ));
        assertThat(result.isTruncated(), equalTo(false));
    }

    @Test
    public void streamCombine() {
        LabelStats result = IntStream.range(0, 6000)
                .parallel()
                .mapToObj(index -> "solomon-metabase-" + index)
                .collect(LabelStatsCollectors.toLabelStats());

        assertThat(result.getCount(), equalTo(6000));
        assertThat(result.getValues(), allOf(
                iterableWithSize(6000),
                hasItem("solomon-metabase-1"),
                hasItem("solomon-metabase-3000"),
                hasItem("solomon-metabase-5999")
        ));
    }

    @Test
    public void tooSmallForLimiting() {
        LabelStats result = LabelStats.create();
        result.add("one");
        result.add("two");
        result.add("tree");
        result.limit(10);

        assertThat(result.isTruncated(), equalTo(false));
        assertThat(result.getCount(), equalTo(3));
        assertThat(result.getValues(), allOf(hasItem("one"), hasItem("two"), hasItem("tree")));
    }

    @Test
    public void limiting() {
        LabelStats result = LabelStats.create();
        result.add("a");
        result.add("b");
        result.add("d");
        result.add("c");
        result.limit(3);

        assertThat(result.isTruncated(), equalTo(true));
        assertThat(result.getCount(), equalTo(4));
        assertThat(result.getValues(), allOf(hasItem("a"), hasItem("b"), hasItem("c"), not(hasItem("d"))));
    }

    @Test
    public void limitingStable() {
        List<String> source = Arrays.asList("10", "50", "100", "500", "1000", "5000", "15000", "inf");
        Collections.shuffle(source);

        LabelStats result = LabelStats.create();
        source.forEach(result::add);
        result.limit(3);

        assertThat(result.isTruncated(), equalTo(true));
        assertThat(result.getCount(), equalTo(8));
        assertThat(result.getValues(), allOf(iterableWithSize(3), hasItem("10"), hasItem("50"), hasItem("100")));
    }

    @Test
    public void filterInvalid() {
        LabelStats stats = LabelStats.create();
        stats.add("valid1");
        stats.add("valid2");
        stats.add("valid3");
        stats.add("{{invalid\n1}}");
        stats.add("{{invalid\n2}}");

        stats.filter(LabelValidationFilter.INVALID_ONLY);
        assertThat(stats.isTruncated(), equalTo(false));
        assertThat(stats.getCount(), equalTo(5));
        assertThat(stats.getValues(), allOf(
            iterableWithSize(2),
            hasItem("{{invalid\n1}}"),
            hasItem("{{invalid\n2}}")
        ));
    }

    @Test
    public void filterNoLimiting() {
        LabelStats stats = LabelStats.create();
        stats.add("my.sensor.elapsedTimeMillis");
        stats.add("jvm.memory.init");
        stats.add("jvm.memory.used");
        stats.add("jvm.memory.committed");
        stats.add("jvm.memory.max");
        stats.add("freeMemory");
        stats.add("diskUsage");
        stats.add("requestStarted");
        stats.add("requestCompleted");

        stats.filter("memory");
        assertThat(stats.isTruncated(), equalTo(false));
        assertThat(stats.getCount(), equalTo(9));
        assertThat(stats.getValues(), allOf(
                iterableWithSize(5),
                hasItem("jvm.memory.init"),
                hasItem("jvm.memory.used"),
                hasItem("jvm.memory.committed"),
                hasItem("jvm.memory.max"),
                hasItem("freeMemory")
        ));
    }

    @Test
    public void filterLimiting() {
        LabelStats stats = LabelStats.create();
        stats.add("my.sensor.elapsedTimeMillis");
        stats.add("jvm.memory.init");
        stats.add("jvm.memory.used");
        stats.add("jvm.memory.committed");
        stats.add("jvm.memory.max");
        stats.add("diskUsage");
        stats.add("requestStarted");
        stats.add("requestCompleted");

        stats.filter("memory");
        stats.limit(2);
        assertThat(stats.isTruncated(), equalTo(true));
        assertThat(stats.getCount(), equalTo(8));
        assertThat(stats.getValues(), allOf(
                iterableWithSize(2),
                hasItem("jvm.memory.committed"),
                hasItem("jvm.memory.init")
        ));
    }

    @Test
    public void filterNoTruncated() {
        LabelStats stats = LabelStats.create();
        stats.add("my.sensor.elapsedTimeMillis");
        stats.add("jvm.memory.init");
        stats.add("jvm.memory.used");
        stats.add("jvm.memory.committed");
        stats.add("jvm.memory.max");
        stats.add("diskUsage");
        stats.add("requestStarted");
        stats.add("requestCompleted");

        stats.filter("request");
        stats.limit(3);
        assertThat(stats.isTruncated(), equalTo(false));
        assertThat(stats.getCount(), equalTo(8));
        assertThat(stats.getValues(), allOf(
                iterableWithSize(2),
                hasItem("requestStarted"),
                hasItem("requestCompleted")
        ));
    }

    @Test
    public void filterLimitingOnSmall() {
        LabelStats stats = LabelStats.create();
        stats.add("jvm.memory.used");
        stats.add("jvm.memory.max");

        stats.filter("jvm");
        stats.limit(10);
        assertThat(stats.isTruncated(), equalTo(false));
        assertThat(stats.getCount(), equalTo(2));
        assertThat(stats.getValues(), allOf(
                iterableWithSize(2),
                hasItem("jvm.memory.used"),
                hasItem("jvm.memory.max")
        ));
    }

    @Test
    public void skipFiltering() {
        LabelStats stats = LabelStats.create();
        stats.add("jvm.memory.used");
        stats.add("jvm.memory.max");

        stats.filter("");
        assertThat(stats.isTruncated(), equalTo(false));
        assertThat(stats.getCount(), equalTo(2));
        assertThat(stats.getValues(), allOf(
                iterableWithSize(2),
                hasItem("jvm.memory.used"),
                hasItem("jvm.memory.max")
        ));
    }

    @Test
    public void skipLimiting() {
        LabelStats stats = LabelStats.create();
        stats.add("jvm.memory.used");
        stats.add("jvm.memory.max");

        stats.limit(0);
        assertThat(stats.isTruncated(), equalTo(false));
        assertThat(stats.getCount(), equalTo(2));
        assertThat(stats.getValues(), allOf(
                iterableWithSize(2),
                hasItem("jvm.memory.used"),
                hasItem("jvm.memory.max")
        ));
    }
}
