package ru.yandex.solomon.util.labelStats;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.stream.IntStream;
import java.util.stream.Stream;

import org.junit.Test;

import ru.yandex.monlib.metrics.labels.Labels;
import ru.yandex.monlib.metrics.labels.validate.LabelValidationFilter;
import ru.yandex.solomon.metabase.api.protobuf.TLabelValues;
import ru.yandex.solomon.metabase.api.protobuf.TLabelValuesResponse;

import static org.hamcrest.Matchers.allOf;
import static org.hamcrest.Matchers.emptyIterable;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.hasItem;
import static org.hamcrest.Matchers.iterableWithSize;
import static org.hamcrest.Matchers.nullValue;
import static org.junit.Assert.assertThat;

/**
 * @author Oleg Baryshnikov
 */
public class LabelValuesStatsTest {

    private final LabelValuesStats first = new LabelValuesStatsBuilder()
        .setMetricsCount(20)
        .addLabelStats("cluster", new LabelStatsBuilder()
            .addValues("cluster")
            .setMetricsCount(20)
            .build())
        .addLabelStats("service", new LabelStatsBuilder()
            .addValues("firstService")
            .setMetricsCount(20)
            .build())
        .addLabelStats("host", new LabelStatsBuilder()
            .addAllValues("cluster", "Man", "Myt", "Sas", "Vla")
            .setMetricsCount(20)
            .build())
        .addLabelStats("path", new LabelStatsBuilder()
            .addValues("/MemoryInfo/HeapMemoryInfo/Used")
            .addValues("/MemoryInfo/HeapMemoryInfo/Init")
            .addValues("/MemoryInfo/HeapMemoryInfo/Total")
            .addValues("/estimateRequestSensorsSizeInMegabytes")
            .setMetricsCount(20)
            .build())
        .addLabelStats("bin", new LabelStatsBuilder()
            .addAllValues("100", "200", "500", "1000")
            .setMetricsCount(4)
            .build())
        .build();

    private final LabelValuesStats second = new LabelValuesStatsBuilder()
        .setMetricsCount(30)
        .addLabelStats("cluster", new LabelStatsBuilder()
            .addValues("cluster")
            .setMetricsCount(30)
            .build())
        .addLabelStats("service", new LabelStatsBuilder()
            .addValues("secondService")
            .setMetricsCount(30)
            .build())
        .addLabelStats("host", new LabelStatsBuilder()
            .addAllValues("cluster", "Man", "Sas", "Vla")
            .setMetricsCount(30)
            .build())
        .addLabelStats("sensor", new LabelStatsBuilder()
            .addValues("jvm.memory.used")
            .addValues("jvm.memory.init")
            .addValues("jvm.memory.total")
            .addValues("estimateRequestSensorsSizeInMegabytes")
            .setMetricsCount(30)
            .build())
        .addLabelStats("bin", new LabelStatsBuilder()
            .addAllValues("100", "300", "1000", "3000", "10000")
            .setMetricsCount(5)
            .build())
        .build();

    private final LabelValuesStats expectedMerged = new LabelValuesStatsBuilder()
        .setMetricsCount(50)
        .addLabelStats("cluster", new LabelStatsBuilder()
            .addValues("cluster")
            .setMetricsCount(50)
            .build())
        .addLabelStats("service", new LabelStatsBuilder()
            .addAllValues("firstService", "secondService")
            .setMetricsCount(50)
            .build())
        .addLabelStats("host", new LabelStatsBuilder()
            .addAllValues("cluster", "Man", "Myt", "Sas", "Vla")
            .setMetricsCount(50)
            .build())
        .addLabelStats("path", new LabelStatsBuilder()
            .addValues("/MemoryInfo/HeapMemoryInfo/Used")
            .addValues("/MemoryInfo/HeapMemoryInfo/Init")
            .addValues("/MemoryInfo/HeapMemoryInfo/Total")
            .addValues("/estimateRequestSensorsSizeInMegabytes")
            .setMetricsCount(20)
            .build())
        .addLabelStats("sensor", new LabelStatsBuilder()
            .addValues("jvm.memory.used")
            .addValues("jvm.memory.init")
            .addValues("jvm.memory.total")
            .addValues("estimateRequestSensorsSizeInMegabytes")
            .setMetricsCount(30)
            .build())
        .addLabelStats("bin", new LabelStatsBuilder()
            .addAllValues("100", "200", "300", "500", "1000", "10000", "3000")
            .setMetricsCount(9)
            .build())
        .build();

    @Test
    public void mergeStats() {
        LabelValuesStats actualMerged = Stream.of(first, second)
                .collect(LabelValuesStats::new, LabelValuesStats::combine, LabelValuesStats::combine);

        assertThat(actualMerged, equalTo(expectedMerged));
    }

    @Test
    public void buildWithSliceOptions() {
        LabelValuesStats result = new LabelValuesStats();
        result.combine(first);
        result.limit(2);

        Map<String, TLabelValues> labelValuesMap = toLabelValuesMap(LabelStatsConverter.toProto(result));

        TLabelValues hostValues = labelValuesMap.get("host");
        assertThat(hostValues.getAbsent(), equalTo(false));
        assertThat(hostValues.getValuesList(), iterableWithSize(2));
        assertThat(hostValues.getTruncated(), equalTo(true));
    }

    @Test
    public void buildWithTextAndLimit() {
        LabelValuesStats result = new LabelValuesStats();
        result.combine(first);
        result.filter("a");
        result.limit(2);

        TLabelValuesResponse response = LabelStatsConverter.toProto(result);

        Map<String, TLabelValues> labelValuesMap = toLabelValuesMap(response);

        TLabelValues hostValues = labelValuesMap.get("host");
        assertThat(hostValues.getAbsent(), equalTo(false));
        assertThat(hostValues.getValuesList(), iterableWithSize(2));
        assertThat(hostValues.getTruncated(), equalTo(true));
    }

    @Test
    public void buildWithShardLimits() {
        LabelValuesStats result = new LabelValuesStats();
        result.combine(expectedMerged);
        result.filter("");
        result.limit(1);

        TLabelValuesResponse response = LabelStatsConverter.toProto(result);

        Map<String, TLabelValues> labelValuesMap = toLabelValuesMap(response);

        TLabelValues serviceValues = labelValuesMap.get("service");
        assertThat(serviceValues.getValuesList(), iterableWithSize(1));
        assertThat(serviceValues.getTruncated(), equalTo(true));
    }

    @Test
    public void buildWithoutShardLimits() {
        LabelValuesStats result = new LabelValuesStats();
        result.combine(expectedMerged);
        result.filter("");

        TLabelValuesResponse response = LabelStatsConverter.toProto(result);

        Map<String, TLabelValues> labelValuesMap = toLabelValuesMap(response);

        TLabelValues serviceValues = labelValuesMap.get("service");
        assertThat(serviceValues.getValuesList(), iterableWithSize(2));
        assertThat(serviceValues.getTruncated(), equalTo(false));
    }

    @Test
    public void buildWithTextSearchForKey() {
        LabelValuesStats result = new LabelValuesStats();
        result.combine(first);
        result.filter("host");
        result.limit(0);

        TLabelValuesResponse response = LabelStatsConverter.toProto(result);
        Map<String, TLabelValues> labelValuesMap = toLabelValuesMap(response);

        TLabelValues host = labelValuesMap.get("host");
        assertThat(host.getAbsent(), equalTo(false));
        assertThat(host.getMetricCount(), equalTo(20));
        assertThat(host.getValuesList(), iterableWithSize(5));
        assertThat(host.getTruncated(), equalTo(false));

        TLabelValues path = labelValuesMap.get("path");
        assertThat(path.getAbsent(), equalTo(false));
        assertThat(path.getMetricCount(), equalTo(20));
        assertThat(path.getValuesList(), iterableWithSize(0));
        assertThat(path.getTruncated(), equalTo(false));

        TLabelValues bin = labelValuesMap.get("bin");
        assertThat(bin.getAbsent(), equalTo(true));
        assertThat(bin.getMetricCount(), equalTo(4));
        assertThat(bin.getValuesList(), iterableWithSize(0));
        assertThat(bin.getTruncated(), equalTo(false));

    }

    @Test
    public void buildWithTextSearchForValues() {
        LabelValuesStats result = new LabelValuesStats();
        result.combine(first);
        result.filter("100");
        result.limit(0);

        TLabelValuesResponse response = LabelStatsConverter.toProto(result);
        Map<String, TLabelValues> labelValuesMap = toLabelValuesMap(response);

        TLabelValues host = labelValuesMap.get("host");
        assertThat(host.getAbsent(), equalTo(false));
        assertThat(host.getMetricCount(), equalTo(20));
        assertThat(host.getValuesList(), iterableWithSize(0));
        assertThat(host.getTruncated(), equalTo(false));

        TLabelValues path = labelValuesMap.get("path");
        assertThat(path.getAbsent(), equalTo(false));
        assertThat(path.getMetricCount(), equalTo(20));
        assertThat(path.getValuesList(), iterableWithSize(0));
        assertThat(path.getTruncated(), equalTo(false));

        TLabelValues bin = labelValuesMap.get("bin");
        assertThat(bin.getAbsent(), equalTo(true));
        assertThat(bin.getMetricCount(), equalTo(4));
        assertThat(bin.getValuesList(), iterableWithSize(2));
        assertThat(bin.getTruncated(), equalTo(false));
    }

    @Test
    public void addLabels() {
        Set<String> filter = Collections.emptySet();
        LabelValuesStats stats = new LabelValuesStats();
        stats.add(Labels.of("sensor", "jvm.memory.init", "host", "test"), filter);
        stats.add(Labels.of("sensor", "jvm.memory.used", "host", "test"), filter);
        stats.add(Labels.of("sensor", "jvm.memory.max", "host", "test"), filter);
        stats.add(Labels.of("sensor", "jvm.memory.committed", "host", "test"), filter);
        stats.add(Labels.of("sensor", "version"), filter);

        assertThat(stats.getMetricsCount(), equalTo(5));

        LabelStats metric = stats.getStatsByLabelKey().get("sensor");
        assertThat(metric.getCount(), equalTo(5));
        assertThat(metric.isTruncated(), equalTo(false));
        assertThat(metric.getValues(), allOf(
                iterableWithSize(5),
                hasItem("jvm.memory.init"),
                hasItem("jvm.memory.used"),
                hasItem("jvm.memory.max"),
                hasItem("jvm.memory.committed"),
                hasItem("version")));

        LabelStats host = stats.getStatsByLabelKey().get("host");
        assertThat(host.getCount(), equalTo(4));
        assertThat(host.isTruncated(), equalTo(false));
        assertThat(host.getValues(), allOf(
                iterableWithSize(1),
                hasItem("test")));
    }

    @Test
    public void addFiltered() {
        Set<String> filter = Collections.singleton("host");
        LabelValuesStats stats = new LabelValuesStats();
        stats.add(Labels.of("sensor", "jvm.memory.init", "host", "test"), filter);
        stats.add(Labels.of("sensor", "jvm.memory.used", "host", "test"), filter);
        stats.add(Labels.of("sensor", "jvm.memory.max", "host", "test"), filter);
        stats.add(Labels.of("sensor", "jvm.memory.committed", "host", "test"), filter);
        stats.add(Labels.of("sensor", "version"), filter);

        assertThat(stats.getMetricsCount(), equalTo(5));

        LabelStats metric = stats.getStatsByLabelKey().get("sensor");
        assertThat(metric, nullValue());

        LabelStats host = stats.getStatsByLabelKey().get("host");
        assertThat(host.getCount(), equalTo(4));
        assertThat(host.isTruncated(), equalTo(false));
        assertThat(host.getValues(), allOf(
                iterableWithSize(1),
                hasItem("test")));
    }

    @Test
    public void streamCollecting() {
        LabelValuesStats stats = IntStream.range(0, 6000)
                .parallel()
                .mapToObj(index -> Labels.of("host", "test-" + index, "sensor", "jvm.memory.used"))
                .collect(LabelStatsCollectors.toLabelValuesStats());
        assertThat(stats.getMetricsCount(), equalTo(6000));

        LabelStats host = stats.getStatsByLabelKey().get("host");
        assertThat(host.isTruncated(), equalTo(false));
        assertThat(host.getCount(), equalTo(6000));
        assertThat(host.getValues(), allOf(iterableWithSize(6000),
                hasItem("test-0"),
                hasItem("test-3000"),
                hasItem("test-5999")
        ));

        LabelStats metric = stats.getStatsByLabelKey().get("sensor");
        assertThat(metric.isTruncated(), equalTo(false));
        assertThat(metric.getCount(), equalTo(6000));
        assertThat(metric.getValues(), allOf(iterableWithSize(1), hasItem("jvm.memory.used")));
    }

    @Test
    public void skipLimit() {
        Set<String> filter = Collections.emptySet();
        LabelValuesStats stats = new LabelValuesStats();
        stats.add(Labels.of("sensor", "jvm.memory.init", "host", "test"), filter);
        stats.add(Labels.of("sensor", "jvm.memory.used", "host", "test"), filter);

        stats.limit(0);

        assertThat(stats.getMetricsCount(), equalTo(2));

        LabelStats metric = stats.getStatsByLabelKey().get("sensor");
        assertThat(metric.getCount(), equalTo(2));
        assertThat(metric.isTruncated(), equalTo(false));
        assertThat(metric.getValues(), allOf(
                iterableWithSize(2),
                hasItem("jvm.memory.init"),
                hasItem("jvm.memory.used")));

        LabelStats host = stats.getStatsByLabelKey().get("host");
        assertThat(host.getCount(), equalTo(2));
        assertThat(host.isTruncated(), equalTo(false));
        assertThat(host.getValues(), allOf(
                iterableWithSize(1),
                hasItem("test")));
    }

    @Test
    public void limitOnSmall() {
        Set<String> filter = Collections.emptySet();
        LabelValuesStats stats = new LabelValuesStats();
        stats.add(Labels.of("sensor", "jvm.memory.init", "host", "test"), filter);
        stats.add(Labels.of("sensor", "jvm.memory.used", "host", "test"), filter);

        stats.limit(10);

        assertThat(stats.getMetricsCount(), equalTo(2));

        LabelStats metric = stats.getStatsByLabelKey().get("sensor");
        assertThat(metric.getCount(), equalTo(2));
        assertThat(metric.isTruncated(), equalTo(false));
        assertThat(metric.getValues(), allOf(
                iterableWithSize(2),
                hasItem("jvm.memory.init"),
                hasItem("jvm.memory.used")));

        LabelStats host = stats.getStatsByLabelKey().get("host");
        assertThat(host.getCount(), equalTo(2));
        assertThat(host.isTruncated(), equalTo(false));
        assertThat(host.getValues(), allOf(
                iterableWithSize(1),
                hasItem("test")));
    }

    @Test
    public void limitForEachLabel() {
        LabelValuesStats stats = IntStream.range(0, 500)
                .parallel()
                .mapToObj(index -> Labels.of("code", String.valueOf(index), "sensor", "requestCompleted"))
                .collect(LabelStatsCollectors.toLabelValuesStats());

        stats.limit(3);
        assertThat(stats.getMetricsCount(), equalTo(500));

        LabelStats metric = stats.getStatsByLabelKey().get("sensor");
        assertThat(metric.getCount(), equalTo(500));
        assertThat(metric.isTruncated(), equalTo(false));
        assertThat(metric.getValues(), allOf(
                iterableWithSize(1),
                hasItem("requestCompleted")));

        LabelStats code = stats.getStatsByLabelKey().get("code");
        assertThat(code.getCount(), equalTo(500));
        assertThat(code.isTruncated(), equalTo(true));
        assertThat(code.getValues(), allOf(
                iterableWithSize(3),
                hasItem("0"),
                hasItem("1"),
                hasItem("2")));
    }

    @Test
    public void skipFilter() {
        Set<String> filter = Collections.emptySet();
        LabelValuesStats stats = new LabelValuesStats();
        stats.add(Labels.of("sensor", "jvm.memory.init", "host", "test"), filter);
        stats.add(Labels.of("sensor", "jvm.memory.used", "host", "test"), filter);

        stats.filter("");
        assertThat(stats.getMetricsCount(), equalTo(2));

        LabelStats metric = stats.getStatsByLabelKey().get("sensor");
        assertThat(metric.getCount(), equalTo(2));
        assertThat(metric.isTruncated(), equalTo(false));
        assertThat(metric.getValues(), allOf(
                iterableWithSize(2),
                hasItem("jvm.memory.init"),
                hasItem("jvm.memory.used")));

        LabelStats host = stats.getStatsByLabelKey().get("host");
        assertThat(host.getCount(), equalTo(2));
        assertThat(host.isTruncated(), equalTo(false));
        assertThat(host.getValues(), allOf(
                iterableWithSize(1),
                hasItem("test")));
    }

    @Test
    public void filterByValue() {
        Set<String> filter = Collections.emptySet();
        LabelValuesStats stats = new LabelValuesStats();
        stats.add(Labels.of("sensor", "jvm.memory.used", "host", "test"), filter);
        stats.add(Labels.of("sensor", "freeSpace", "host", "test"), filter);
        stats.add(Labels.of("sensor", "useTime", "host", "test"), filter);
        stats.add(Labels.of("sensor", "freeTime", "host", "test"), filter);

        stats.filter("memory");
        assertThat(stats.getMetricsCount(), equalTo(4));

        LabelStats metric = stats.getStatsByLabelKey().get("sensor");
        assertThat(metric.getCount(), equalTo(4));
        assertThat(metric.isTruncated(), equalTo(false));
        assertThat(metric.getValues(), allOf(
                iterableWithSize(1),
                hasItem("jvm.memory.used")));

        LabelStats host = stats.getStatsByLabelKey().get("host");
        assertThat(host.getCount(), equalTo(4));
        assertThat(host.isTruncated(), equalTo(false));
        assertThat(host.getValues(), emptyIterable());
    }

    @Test
    public void filterByKey() {
        Set<String> filter = Collections.emptySet();
        LabelValuesStats stats = new LabelValuesStats();
        stats.add(Labels.of("sensor", "jvm.memory.used", "host", "test"), filter);
        stats.add(Labels.of("sensor", "freeSpace", "host", "test"), filter);
        stats.add(Labels.of("sensor", "useTime", "host", "test"), filter);
        stats.add(Labels.of("sensor", "freeTime", "host", "test"), filter);

        stats.filter("host");
        assertThat(stats.getMetricsCount(), equalTo(4));

        LabelStats metric = stats.getStatsByLabelKey().get("sensor");
        assertThat(metric.getCount(), equalTo(4));
        assertThat(metric.isTruncated(), equalTo(false));
        assertThat(metric.getValues(), emptyIterable());

        LabelStats host = stats.getStatsByLabelKey().get("host");
        assertThat(host.getCount(), equalTo(4));
        assertThat(host.isTruncated(), equalTo(false));
        assertThat(host.getValues(), allOf(iterableWithSize(1), hasItem("test")));
    }

    @Test
    public void filterInvalidKeys() {
        LabelValuesStats stats = new LabelValuesStatsBuilder()
            .addLabelStats("stream-name", new LabelStatsBuilder()
                .addAllValues("value1", "value2", "value3")
                .setMetricsCount(9)
                .build()
            )
            .addLabelStats("host", new LabelStatsBuilder()
                .addAllValues("cluster", "Vla", "Sas")
                .setMetricsCount(9)
                .build())
            .setMetricsCount(9)
            .build();

        stats.filter(LabelValidationFilter.INVALID_ONLY);

        assertThat(stats.getMetricsCount(), equalTo(9));

        LabelStats streamName = stats.getStatsByLabelKey().get("stream-name");
        assertThat(streamName.getCount(), equalTo(9));
        assertThat(streamName.getValues(), emptyIterable());

        LabelStats host = stats.getStatsByLabelKey().get("host");
        assertThat(host.getCount(), equalTo(9));
        assertThat(host.getValues(), emptyIterable());
    }


    @Test
    public void filterInvalidValues() {
        LabelValuesStats stats = new LabelValuesStatsBuilder()
            .addLabelStats("sensor", new LabelStatsBuilder()
                .addAllValues("value", "\n100", "\n200")
                .setMetricsCount(3)
                .build()
            )
            .setMetricsCount(3)
            .build();

        stats.filter(LabelValidationFilter.INVALID_ONLY);

        assertThat(stats.getMetricsCount(), equalTo(3));

        LabelStats metric = stats.getStatsByLabelKey().get("sensor");
        assertThat(metric.getCount(), equalTo(3));
        assertThat(metric.getValues(), allOf(iterableWithSize(2), hasItem("\n100"), hasItem("\n200")));
    }

    private static Map<String, TLabelValues> toLabelValuesMap(TLabelValuesResponse response) {
        Map<String, TLabelValues> labelValuesMap = new HashMap<>(response.getValuesCount());

        for (TLabelValues labelValues : response.getValuesList()) {
            labelValuesMap.put(labelValues.getName(), labelValues);
        }

        return labelValuesMap;
    }

    private static class LabelValuesStatsBuilder {

        private int metricsCount = 0;
        private Map<String, LabelStats> statsByLabelKey = new HashMap<>();

        LabelValuesStatsBuilder setMetricsCount(int metricsCount) {
            this.metricsCount = metricsCount;
            return this;
        }

        LabelValuesStatsBuilder addLabelStats(String name, LabelStats stats) {
            this.statsByLabelKey.put(name, stats);
            return this;
        }

        public LabelValuesStats build() {
            return new LabelValuesStats(statsByLabelKey, metricsCount);
        }
    }

    private static class LabelStatsBuilder {
        private Set<String> values = new HashSet<>();
        private int metricsCount = 0;
        boolean truncated = false;

        LabelStatsBuilder setMetricsCount(int metricsCount) {
            this.metricsCount = metricsCount;
            return this;
        }

        LabelStatsBuilder addValues(String value) {
            this.values.add(value);
            return this;
        }

        LabelStatsBuilder setTruncated(boolean truncated) {
            this.truncated = truncated;
            return this;
        }

        LabelStatsBuilder addAllValues(String... values) {
            this.values.addAll(Arrays.asList(values));
            return this;
        }

        public LabelStats build() {
            return new LabelStats(values, metricsCount, truncated);
        }
    }
}
