package ru.yandex.solomon.util.net;

import java.util.Arrays;
import java.util.List;

import org.junit.Assert;
import org.junit.BeforeClass;
import org.junit.Test;
import org.junit.experimental.runners.Enclosed;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

/**
 * @author alexlovkov
 */
@RunWith(Enclosed.class)
public class DefaultNetworkValidatorTest {

    public static class TestValidIps {

        static NetworkValidator validator;

        @BeforeClass
        public static void init() {
            List<String> ipv4 = Arrays.asList("127.0.0.0/28", "123.0.0.0/15");
            List<String> ipv6 = Arrays.asList("bbbb:10f:d000::/64", "ffff:10f:d000::/125");
            validator = new DefaultNetworkValidator(ipv4, ipv6);
        }

        @Test
        public void successValidate() {
            validator.validateNetwork("127.0.0.1");
            validator.validateNetwork("127.0.0.2");
            validator.validateNetwork("127.0.0.14");
            validator.validateNetwork("127.0.0.0/30");
            validator.validateNetwork("127.0.0.0/29");

            validator.validateNetwork("123.0.15.166");
            validator.validateNetwork("bbbb:10f:d000:0:0:0:0:1");
            validator.validateNetwork("bbbb:10f:d000:0:0:0:100:100");
            validator.validateNetwork("ffff:10f:d000:0:0:0:0:3");

            validator.validateNetwork("bbbb:10f:d000::/123");
            validator.validateNetwork("ffff:10f:d000::/126");
        }

        @Test(expected = IllegalArgumentException.class)
        public void tooManyIpv4() {
            validator.validateNetwork("123.0.0.0/12");
        }

        @Test(expected = IllegalArgumentException.class)
        public void tooManyIpv6() {
            validator.validateNetwork("ffff:10f:d000::/15");
        }

        @Test
        public void testIpv4Text() {
            try {
                validator.validateNetwork("127.0.0.0");
            } catch (Exception e) {
                Assert.assertTrue(e.getMessage().contains("ipv4"));
            }
            try {
                validator.validateNetwork("127.0.0.0/26");
            } catch (Exception e) {
                Assert.assertTrue(e.getMessage().contains("ipv4"));
            }
        }

        @Test
        public void testIpv6Text() {
            try {
                validator.validateNetwork("bbbb:10f:d::");
            } catch (Exception e) {
                Assert.assertTrue(e.getMessage().contains("ipv6"));
            }
            try {
                validator.validateNetwork("ffff:10f:d000::/123");
            } catch (Exception e) {
                Assert.assertTrue(e.getMessage().contains("ipv6"));
            }
        }
    }


    @RunWith(Parameterized.class)
    public static class TestInvalidIps {

        static NetworkValidator validator;
        String notInsideNetwork;

        public TestInvalidIps(String notInsideNetwork) {
            this.notInsideNetwork = notInsideNetwork;
        }

        @Parameterized.Parameters(name = "{0}")
        public static List<String> invalidIps() {
            return Arrays.asList(
                "127.0.0.0",
                "127.0.0.3",
                "127.0.1.0",
                "127.0.0.15",
                "127.0.0.0/25",
                "bbbb:10f:d000:0:0:0:0:0",
                "bbbb:10f:d::",
                "bbbb:10f:d:0:0::",
                "ffff:10f:d000:0:0:0:0:8",
                "ffff:10f:d000:0:0:0:0:9",
                "ffff:10f:e::",
                "ffff:10f:d000::/123"
            );
        }

        @BeforeClass
        public static void init() {
            List<String> ipv4 = Arrays.asList("127.0.0.0/30", "123.0.0.0/15");
            List<String> ipv6 = Arrays.asList("bbbb:10f:d000::/64", "ffff:10f:d000::/125");
            validator = new DefaultNetworkValidator(ipv4, ipv6);
        }

        @Test(expected = IllegalArgumentException.class)
        public void testInvalidIp() {
            validator.validateNetwork(notInsideNetwork);
        }

    }
}
