package ru.yandex.solomon.util.protobuf;

import java.util.Arrays;

import com.google.protobuf.ByteString;
import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufUtil;
import io.netty.buffer.CompositeByteBuf;
import io.netty.buffer.Unpooled;
import org.junit.Assert;
import org.junit.Test;


/**
 * @author Sergey Polovko
 */
public class ByteStringsTest {

    private static final byte[] DATA = {
        (byte) 0x00, (byte) 0x11, (byte) 0x22, (byte) 0x33, (byte) 0x44, (byte) 0x55, (byte) 0x66, (byte) 0x77,
        (byte) 0x88, (byte) 0x99, (byte) 0xaa, (byte) 0xbb, (byte) 0xcc, (byte) 0xdd, (byte) 0xee, (byte) 0xff,
    };

    @Test
    public void emptyBuffer() {
        ByteString str = ByteStrings.fromByteBuf(Unpooled.EMPTY_BUFFER);
        Assert.assertSame(ByteString.EMPTY, str);
    }

    @Test
    public void directBuffer() {
        ByteBuf buf = Unpooled.directBuffer();
        try {
            buf.writeBytes(DATA);
            ByteString str = ByteStrings.fromByteBuf(buf);
            byte[] bytes = str.toByteArray();
            Assert.assertArrayEquals(DATA, bytes);
        } finally {
            buf.release();
        }
    }

    @Test
    public void heapBuffer() {
        ByteBuf buf = Unpooled.copiedBuffer(DATA);
        ByteString str = ByteStrings.fromByteBuf(buf);
        byte[] bytes = str.toByteArray();
        Assert.assertArrayEquals(DATA, bytes);
    }

    @Test
    public void compositeBuffer() {
        CompositeByteBuf buf = Unpooled.compositeBuffer();
        buf.addComponent(true, Unpooled.copiedBuffer(DATA));
        buf.addComponent(true, Unpooled.copiedBuffer(DATA));

        ByteString str = ByteStrings.fromByteBuf(buf);
        byte[] bytes = str.toByteArray();

        byte[] expected = new byte[DATA.length * 2];
        System.arraycopy(DATA, 0, expected, 0, DATA.length);
        System.arraycopy(DATA, 0, expected, DATA.length, DATA.length);

        Assert.assertArrayEquals(expected, bytes);
    }

    @Test
    public void emptyString() {
        ByteBuf buf = ByteStrings.toByteBuf(ByteString.EMPTY);
        Assert.assertSame(Unpooled.EMPTY_BUFFER, buf);
    }

    @Test
    public void literalString() {
        ByteBuf buf = ByteStrings.toByteBuf(ByteString.copyFrom(DATA));
        byte[] bytes = ByteBufUtil.getBytes(buf);
        Assert.assertArrayEquals(DATA, bytes);
    }

    @Test
    public void boundedString() {
        ByteString str = ByteString.copyFrom(DATA)
            .substring(0, 8);

        ByteBuf buf = ByteStrings.toByteBuf(str);
        byte[] bytes = ByteBufUtil.getBytes(buf);

        Assert.assertArrayEquals(Arrays.copyOf(DATA, 8), bytes);
    }
}
