package ru.yandex.solomon.util.protobuf;

import java.io.IOException;

import com.google.common.base.Charsets;
import com.google.common.io.Resources;
import com.google.protobuf.TextFormat;
import org.junit.Assert;
import org.junit.Test;

import ru.yandex.solomon.common.StringPool.Compression;

/**
 * @author Sergey Polovko
 */
public class StringPoolTest {

    @Test
    public void fromProto() throws IOException {
        String data = Resources.toString(StringPoolTest.class.getResource("test_pool.txt"), Charsets.UTF_8);
        StringPool strings = StringPool.fromProto(TextFormat.parse(data, ru.yandex.solomon.common.StringPool.class));

        String[] expected = {
                "",
                "cluster",
                "production",
                "project",
                "solomon",
                "service",
                "coremon",
                "host",
                "solomon-fetcher-sas-001",
                "sensor",
                "jvm.threadPool.activeThreads",
                "threadPool",
                "Http",
                "CpuHighPriority",
                "MetabasePool",
                "ParsingThreadPool",
                "Scheduler",
                "CommonFJP",
                "CpuLowPriority",
        };

        // -1 because an empty string is not stored in the pool
        Assert.assertEquals(expected.length - 1, strings.size());

        for (int i = 0; i <= strings.size(); i++) {
            Assert.assertEquals("not equals string at #" + i, expected[i], strings.get(i));
        }
    }

    @Test
    public void builderPut() {
        var builder = StringPool.newBuilder();

        int aIdx = builder.put("a");
        Assert.assertEquals(1, aIdx);
        Assert.assertEquals(1, builder.size());

        int bIdx = builder.put("b");
        Assert.assertEquals(2, bIdx);
        Assert.assertEquals(2, builder.size());

        int cIdx = builder.put("c");
        Assert.assertEquals(3, cIdx);
        Assert.assertEquals(3, builder.size());

        Assert.assertEquals(cIdx, builder.put("c"));
        Assert.assertEquals(bIdx, builder.put("b"));
        Assert.assertEquals(aIdx, builder.put("a"));

        Assert.assertEquals(3, builder.size());
    }

    @Test
    public void build() {
        // empty
        {
            var builder = StringPool.newBuilder();
            var strings = builder.build();
            Assert.assertEquals(0, strings.size());
        }

        // single
        {
            var builder = StringPool.newBuilder();
            builder.put("one");

            var strings = builder.build();
            Assert.assertEquals(1, strings.size());
            Assert.assertEquals("", strings.get(0));
            Assert.assertEquals("one", strings.get(1));
        }

        var builder = StringPool.newBuilder();
        builder.put("the");
        builder.put("quick");
        builder.put("brown");
        builder.put("fox");

        StringPool strings = builder.build();
        Assert.assertEquals(4, strings.size());

        Assert.assertEquals("", strings.get(0));
        Assert.assertEquals("the", strings.get(1));
        Assert.assertEquals("quick", strings.get(2));
        Assert.assertEquals("brown", strings.get(3));
        Assert.assertEquals("fox", strings.get(4));
    }

    @Test
    public void buildProto() {
        // empty
        {
            var builder = StringPool.newBuilder();
            var strings = builder.buildProto(Compression.COMPRESSION_UNSPECIFIED);
            Assert.assertSame(ru.yandex.solomon.common.StringPool.getDefaultInstance(), strings);
        }

        // single
        {
            var builder = StringPool.newBuilder();
            builder.put("one");

            var strings = StringPool.fromProto(builder.buildProto(Compression.COMPRESSION_UNSPECIFIED));
            Assert.assertEquals(1, strings.size());
            Assert.assertEquals("", strings.get(0));
            Assert.assertEquals("one", strings.get(1));
        }

        var builder = StringPool.newBuilder();
        builder.put("the");
        builder.put("quick");
        builder.put("brown");
        builder.put("fox");

        for (Compression compression : Compression.values()) {
            if (compression == Compression.UNRECOGNIZED) {
                continue;
            }

            var proto = builder.buildProto(compression);
            Assert.assertEquals(compression, proto.getCompression());

            var strings = StringPool.fromProto(proto);
            Assert.assertEquals(4, strings.size());

            Assert.assertEquals("", strings.get(0));
            Assert.assertEquals("the", strings.get(1));
            Assert.assertEquals("quick", strings.get(2));
            Assert.assertEquals("brown", strings.get(3));
            Assert.assertEquals("fox", strings.get(4));
        }
    }
}
