package ru.yandex.solomon.util.time;

import java.time.Duration;
import java.time.Instant;

import org.hamcrest.CoreMatchers;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;

import ru.yandex.misc.thread.ThreadLocalTimeout;
import ru.yandex.misc.thread.ThreadLocalTimeoutException;

import static org.hamcrest.CoreMatchers.equalTo;
import static org.junit.Assert.assertThat;

/**
 * @author chebik
 */
public class DeadlineTest {

    private ThreadLocalTimeout.Handle handle;
    private org.joda.time.Instant deadlineRaw;

    @Before
    public void setUp() {
        deadlineRaw = org.joda.time.Instant.now().plus(org.joda.time.Duration.millis(2000));
        handle = ThreadLocalTimeout.push(deadlineRaw);
    }

    @After
    public void tearDown() {
        handle.popSafely();
    }

    @Test
    public void testBuildFromThreadLocal() {
        Deadline deadline = Deadline.fromThreadLocal("unit_test");

        assertThat(deadline.toJodaInstant(), equalTo(deadlineRaw));
    }

    @Test
    public void shouldReturnEvenIfThreadLocalNotFound() {
        handle.popSafely();

        Deadline deadline = Deadline.fromThreadLocal("unit_test");
        deadline.check();
    }

    @Test
    public void shouldThrowOnCheckIfDeadlineExceeded() throws InterruptedException {
        Deadline deadline = Deadline.from("unit_test", Instant.now().plus(Duration.ofMillis(100)));

        Thread.sleep(300);

        try {

            deadline.check();
            Assert.fail("Must throw exception");
        } catch (ThreadLocalTimeoutException e) {
            assertThat(e.getMessage(), CoreMatchers.containsString("Operation unit_test timed out"));
        }
    }

    @Test
    public void shouldNotThrowIfDeadlineIsNotExceeded() {
        Deadline deadline = Deadline.from(Instant.now().plus(Duration.ofSeconds(5)));

        deadline.check();
    }

    @Test
    public void shouldNotAffectThreadLocalDeadline() {
        org.joda.time.Instant prevDeadline = ThreadLocalTimeout.deadline().get();

        Deadline deadline = Deadline.fromThreadLocal("unit_test");

        ThreadLocalTimeout.Handle handle = ThreadLocalTimeout.push(deadline.toJodaInstant());
        Assert.assertThat(prevDeadline, equalTo(ThreadLocalTimeout.deadline().get()));
        handle.popSafely();
    }
}
