package ru.yandex.solomon.util.time;

import java.time.Duration;
import java.time.temporal.ChronoUnit;
import java.util.concurrent.TimeUnit;

import org.junit.Test;

import static org.junit.Assert.assertEquals;

/**
 * @author Stepan Koltsov
 */
public class DurationUtilsTest {

    @Test
    public void testToStringUpToMillis() throws Exception {
        assertEquals("0.002", DurationUtils.toStringUpToMillis(Duration.ofMillis(2)));
        assertEquals("20.012", DurationUtils.toStringUpToMillis(Duration.ofMillis(20012)));
        assertEquals("-1.002", DurationUtils.toStringUpToMillis(Duration.ofMillis(1002).negated()));
    }

    @Test
    public void parse() {
        assertEquals(Duration.ofHours(17), DurationUtils.parseDuration("17h").get());
        assertEquals(Duration.ofMinutes(90), DurationUtils.parseDuration("1h30m").get());
        assertEquals(Duration.ofDays(2), DurationUtils.parseDuration("2d").get());
        assertEquals(Duration.ofDays(21), DurationUtils.parseDuration("3w").get());
        assertEquals(Duration.ofDays(22), DurationUtils.parseDuration("3w1d").get());
        assertEquals(Duration.ofSeconds(17), DurationUtils.parseDuration("017s").get());
    }

    @Test
    public void formatDurationMillis() {
        assertEquals("0", DurationUtils.formatDurationMillis(0));
        assertEquals("1h", DurationUtils.formatDurationMillis(TimeUnit.HOURS.toMillis(1)));
        assertEquals("10m", DurationUtils.formatDurationMillis(TimeUnit.MINUTES.toMillis(10)));
        assertEquals("20s", DurationUtils.formatDurationMillis(TimeUnit.SECONDS.toMillis(20)));
        assertEquals("1h30m", DurationUtils.formatDurationMillis(TimeUnit.MINUTES.toMillis(90)));
        assertEquals("1d2h", DurationUtils.formatDurationMillis(TimeUnit.HOURS.toMillis(26)));
    }

    @Test
    public void formatDuration() {
        assertEquals("1h", DurationUtils.formatDuration(Duration.ofHours(1)));
        assertEquals("10m", DurationUtils.formatDuration(Duration.ofMinutes(10)));
        assertEquals("20s", DurationUtils.formatDuration(Duration.ofSeconds(20)));
        assertEquals("0", DurationUtils.formatDuration(Duration.ofMillis(0)));
        assertEquals("1h30m", DurationUtils.formatDuration(Duration.ofMinutes(90)));
        assertEquals("1d2h", DurationUtils.formatDuration(Duration.of(26, ChronoUnit.HOURS)));
    }

    @Test
    public void backoffMaxDelay() {
        assertEquals(30_000L, DurationUtils.backoff(100L, 30_000L, 60));
        assertEquals(30_000L, DurationUtils.backoff(100L, 30_000L, 100500));
        assertEquals(30_000L, DurationUtils.backoff(100L, 30_000L, Integer.MAX_VALUE));
    }

    @Test
    public void backoffDelay() {
        assertEquals(10, DurationUtils.backoff(10, 30_000, 0));
        assertEquals(20, DurationUtils.backoff(10, 30_000, 1));
        assertEquals(40, DurationUtils.backoff(10, 30_000, 2));
        assertEquals(80, DurationUtils.backoff(10, 30_000, 3));
        assertEquals(160, DurationUtils.backoff(10, 30_000, 4));
        assertEquals(320, DurationUtils.backoff(10, 30_000, 5));
        assertEquals(640, DurationUtils.backoff(10, 30_000, 6));
    }
}
