package ru.yandex.solomon.util.time;

import java.time.Duration;
import java.time.Instant;
import java.time.temporal.ChronoUnit;

import org.junit.Assert;
import org.junit.Test;

public class IntervalTest {

    @Test
    public void testCountPointsSimple() throws Exception {
        Instant begin = Instant.ofEpochSecond(100);
        long expected = 1000;
        Duration step = Duration.of(10, ChronoUnit.SECONDS);
        Instant end = begin.plus(step.multipliedBy(expected));
        Interval interval = new Interval(begin, end);
        long found = interval.countPoints(step);
        Assert.assertEquals(expected, found);
    }

    @Test
    public void testCountPointsBeginMoved() throws Exception {
        Instant base = Instant.ofEpochSecond(100);
        long expected = 1000;
        Duration step = Duration.of(10, ChronoUnit.SECONDS);
        Instant begin = base.plus(step.dividedBy(2));
        Instant end = base.plus(step.multipliedBy(expected + 1));
        Interval interval = new Interval(begin, end);
        long found = interval.countPoints(step);
        Assert.assertEquals(expected, found);
    }

    @Test
    public void testCountPointsEndMoved() throws Exception {
        Instant base = Instant.ofEpochSecond(100);
        long expected = 1000;
        Duration step = Duration.of(10, ChronoUnit.SECONDS);
        Instant begin = base;
        Instant end = base.plus(step.multipliedBy(expected - 1)).plus(step.dividedBy(2));
        Interval interval = new Interval(begin, end);
        long found = interval.countPoints(step);
        Assert.assertEquals(expected, found);
    }

    @Test
    public void testCountPointsBothMoved() throws Exception {
        Instant base = Instant.ofEpochSecond(100);
        long expected = 1000;
        Duration step = Duration.of(10, ChronoUnit.SECONDS);
        Instant begin = base.plus(step.dividedBy(10));
        Instant end = base.plus(step.multipliedBy(expected)).plus(step.dividedBy(2));
        Interval interval = new Interval(begin, end);
        long found = interval.countPoints(step);
        Assert.assertEquals(expected, found);
    }

    @Test
    public void testGetPointNumber() throws Exception {
        Instant base = Instant.ofEpochSecond(100);
        long width = 1000;
        Duration step = Duration.of(10, ChronoUnit.SECONDS);
        Instant begin = base.plus(step.dividedBy(10));
        Instant end = base.plus(step.multipliedBy(width)).plus(step.dividedBy(2));
        Interval interval = new Interval(begin, end);
        Assert.assertEquals(0, interval.getPointNumber(base, step));
        Assert.assertEquals(0, interval.getPointNumber(begin, step));
        Assert.assertEquals(width, interval.getPointNumber(end, step));
        Assert.assertEquals(1, interval.getPointNumber(base.plus(step), step));
        Assert.assertEquals(1, interval.getPointNumber(begin.plus(step), step));
    }

    @Test
    public void testContainsOpenClose() throws Exception {
        Interval interval = new Interval(
                Instant.parse("2015-05-19T12:00:00Z"),
                Instant.parse("2015-05-19T14:00:00Z"));
        Assert.assertFalse(interval.containsOpenClose(Instant.parse("2015-05-19T11:00:00Z")));
        Assert.assertTrue(interval.containsOpenClose(Instant.parse("2015-05-19T12:00:00Z")));
        Assert.assertTrue(interval.containsOpenClose(Instant.parse("2015-05-19T13:00:00Z")));
        Assert.assertFalse(interval.containsOpenClose(Instant.parse("2015-05-19T14:00:00Z")));
        Assert.assertFalse(interval.containsOpenClose(Instant.parse("2015-05-19T15:00:00Z")));
    }

    @Test
    public void testExtendBeginInstant() throws Exception {
        Instant end = Instant.parse("2018-01-01T00:00:00Z");

        Interval one = new Interval(Instant.parse("2017-01-01T00:00:00Z"), end);
        Interval two = new Interval(Instant.parse("2016-01-01T00:00:00Z"), end);

        Interval result = Interval.convexHull(one, two);
        Assert.assertEquals(result, two);
    }

    @Test
    public void testExtendEndInstant() throws Exception {
        Instant start = Instant.parse("2018-01-01T00:00:00Z");

        Interval one = new Interval(start, Instant.parse("2018-05-01T00:00:00Z"));
        Interval two = new Interval(start, Instant.parse("2018-03-01T00:00:00Z"));

        Interval result = Interval.convexHull(one, two);
        Assert.assertEquals(result, one);
    }

    @Test
    public void testExtendInterval() throws Exception {
        Interval one = new Interval(Instant.parse("2017-01-01T00:00:00Z"), Instant.parse("2018-05-01T00:00:00Z"));
        Interval two = new Interval(Instant.parse("2018-01-01T00:00:00Z"), Instant.parse("2020-03-01T00:00:00Z"));
        Interval ext = new Interval(Instant.parse("2017-01-01T00:00:00Z"), Instant.parse("2020-03-01T00:00:00Z"));

        Interval result = Interval.convexHull(one, two);
        Assert.assertEquals(ext, result);
    }

    @Test
    public void testExtendYourselfNotAffectInterval() throws Exception {
        Interval interval = new Interval(Instant.parse("2017-01-01T00:00:00Z"), Instant.parse("2018-05-01T00:00:00Z"));

        Interval result = Interval.convexHull(interval, interval);
        Assert.assertEquals(interval, result);
    }
}
