package ru.yandex.solomon.abc;

import java.util.Map;
import java.util.Optional;

import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Bean;
import org.springframework.stereotype.Component;

import ru.yandex.abc.AbcClient;
import ru.yandex.abc.AbcClientOptions;
import ru.yandex.abc.HttpAbcClient;
import ru.yandex.passport.tvmauth.NativeTvmClient;
import ru.yandex.passport.tvmauth.TvmApiSettings;
import ru.yandex.passport.tvmauth.TvmClient;
import ru.yandex.solomon.auth.AuthType;
import ru.yandex.solomon.config.TimeConverter;
import ru.yandex.solomon.config.protobuf.TAbcClientConfig;
import ru.yandex.solomon.config.protobuf.frontend.TAuthConfig;
import ru.yandex.solomon.config.thread.ThreadPoolProvider;
import ru.yandex.solomon.secrets.SecretProvider;
import ru.yandex.solomon.util.SolomonEnv;

/**
 * @author snoop
 * @author Sergey Polovko
 */
@Component
public class AbcClientContext {

    @Bean
    AbcClient abcClient(
            Optional<TAbcClientConfig> configOptional,
            ThreadPoolProvider threads,
            SecretProvider secrets,
            @Qualifier("duty2TvmClient") Optional<TvmClient> tvmClient)
    {
        if (configOptional.isEmpty()) {
            return null;
        }

        var config = configOptional.get();
        var oauthToken = secrets.getSecret(config.getOAuthToken());

        if (SolomonEnv.DEVELOPMENT.isActive() && oauthToken.isEmpty()) {
            // only in local mode oauth token can be empty
            return null;
        }

        var opts = AbcClientOptions.newBuilder()
                .setUrl(config.getUrl())
                .setOauthToken(oauthToken.orElseThrow(() -> new RuntimeException("ABC client's OAuth token is empty")))
                .setHandlerExecutor(threads.getExecutorService(config.getThreadPoolName(), "AbcClient.ThreadPoolName"));

        if (config.hasConnectTimeout()) {
            opts.setConnectTimeout(TimeConverter.protoToDuration(config.getConnectTimeout()));
        }

        if (config.hasRequestTimeout()) {
            opts.setRequestTimeout(TimeConverter.protoToDuration(config.getRequestTimeout()));
        }
        if (config.getDuty2TvmDestinationId() > 0 && tvmClient.isPresent()) {
            opts.setTokenProviderDuty2(() ->
                    AuthType.TvmService.getValuePrefix() + tvmClient.get().getServiceTicketFor(config.getDuty2TvmDestinationId()));
            opts.setTokenHeaderProviderDuty2(AuthType.TvmService::getHeaderName);
        }

        return new HttpAbcClient(opts.build());
    }

    @Bean(name = "duty2TvmClient")
    TvmClient duty2TvmClient(
            TAuthConfig authConfig,
            SecretProvider secrets,
            Optional<TAbcClientConfig> configOptional)
    {
        TAuthConfig.TTvmConfig tvmConfig = authConfig.getTvmConfig();
        Optional<String> clientSecret = secrets.getSecret(tvmConfig.getSecret());
        if (tvmConfig.getSecret().isEmpty()
                || clientSecret.isEmpty()
                || configOptional.isEmpty()
                || configOptional.get().getDuty2TvmDestinationId() <=0)
        {
            return null;
        }
        return new NativeTvmClient(TvmApiSettings.create()
                .setSelfTvmId(tvmConfig.getClientId())
                .enableServiceTicketChecking()
                .enableServiceTicketsFetchOptions(clientSecret.get(), Map.of("duty2", configOptional.get().getDuty2TvmDestinationId())));
    }
}
