package ru.yandex.solomon.abc.validator;

import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;

import com.google.common.collect.ImmutableSet;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Import;
import org.springframework.stereotype.Component;

import ru.yandex.abc.AbcClient;
import ru.yandex.abc.dto.AbcService;
import ru.yandex.misc.concurrent.CompletableFutures;
import ru.yandex.solomon.abc.AbcClientContext;
import ru.yandex.solomon.core.exceptions.BadRequestException;
import ru.yandex.solomon.core.exceptions.NotFoundException;

/**
 * @author snoop
 */
@Component
@Import({ AbcClientContext.class })
public class AbcServiceFieldValidator {

    private static final Set<String> FORBIDDEN_SERVICE_STATES;
    static {
        FORBIDDEN_SERVICE_STATES = ImmutableSet.of("closed", "deleted");
    }

    @Nullable
    private final AbcClient abcClient;

    public AbcServiceFieldValidator(@Autowired Optional<AbcClient> abcClient) {
        this.abcClient = abcClient.orElse(null);
    }

    public CompletableFuture<Void> validate(@Nonnull String abcService, boolean isNew) {
        if (abcClient == null) {
            return CompletableFuture.completedFuture(null);
        }

        if (StringUtils.isBlank(abcService)) {
            throw new BadRequestException("ABC service must not be blank");
        }

        CompletableFuture<Void> promise = new CompletableFuture<>();
        queryService(abcService).whenComplete((r, x) -> {
            if (x != null) {
                promise.completeExceptionally(CompletableFutures.unwrapCompletionException(x));
            } else {
                if (r.isPresent()) {
                    AbcService service = r.get();
                    if (isNew && FORBIDDEN_SERVICE_STATES.contains(service.getState())) {
                        promise.completeExceptionally(serviceStateForbidden(service));
                    } else {
                        promise.complete(null);
                    }
                } else {
                    promise.completeExceptionally(serviceNotFound(abcService));
                }
            }
        });
        return promise;
    }

    private static NotFoundException serviceNotFound(String abcService) {
        return new NotFoundException("ABC service " + abcService + " not found");
    }

    private static BadRequestException serviceStateForbidden(AbcService service) {
        String state = StringUtils.wrap(service.getState(), "'");
        return new BadRequestException("ABC service " + service.getSlug() + " is in state " + state);
    }

    private CompletableFuture<Optional<AbcService>> queryService(String abcService) {
        if (abcClient == null) {
            return CompletableFuture.completedFuture(Optional.empty());
        }
        return abcClient.getService(abcService);
    }
}
