package ru.yandex.abc;

import java.time.Duration;
import java.util.concurrent.Executor;
import java.util.function.Supplier;

import javax.annotation.ParametersAreNonnullByDefault;

/**
 * @author Oleg Baryshnikov
 */
public class AbcClientOptions {

    private final String url;
    private final String oauthToken;
    private final Duration connectTimeout;
    private final Duration requestTimeout;
    private final Executor handlerExecutor;
    private final Supplier<String> tokenProviderDuty2;
    private final Supplier<String> tokenHeaderProviderDuty2;

    private AbcClientOptions(Builder builder) {
        url = builder.url;
        oauthToken = builder.oauthToken;
        connectTimeout = builder.connectTimeout;
        requestTimeout = builder.requestTimeout;
        handlerExecutor = builder.handlerExecutor;
        tokenProviderDuty2 = builder.tokenProviderDuty2;
        tokenHeaderProviderDuty2 = builder.tokenHeaderProviderDuty2;
    }

    public static Builder newBuilder() {
        return new Builder();
    }

    public Supplier<String> getTokenHeaderProviderDuty2() {
        return tokenHeaderProviderDuty2;
    }

    public Supplier<String> getTokenProviderDuty2() {
        return tokenProviderDuty2;
    }

    public String getUrl() {
        return url;
    }

    public String getOauthToken() {
        return oauthToken;
    }

    public Duration getConnectTimeout() {
        return connectTimeout;
    }

    public Duration getRequestTimeout() {
        return requestTimeout;
    }

    public Executor getHandlerExecutor() {
        return handlerExecutor;
    }

    @ParametersAreNonnullByDefault
    public static final class Builder {
        private String url;
        private String oauthToken;
        private Duration connectTimeout = Duration.ofSeconds(5);
        private Duration requestTimeout = Duration.ofSeconds(20);
        private Executor handlerExecutor;
        private Supplier<String> tokenProviderDuty2;
        private Supplier<String> tokenHeaderProviderDuty2;

        public Builder setUrl(String url) {
            this.url = url;
            return this;
        }

        public Builder setOauthToken(String oauthToken) {
            this.oauthToken = oauthToken;
            return this;
        }

        public Builder setTokenHeaderProviderDuty2(Supplier<String> tokenHeaderProviderDuty2) {
            this.tokenHeaderProviderDuty2 = tokenHeaderProviderDuty2;
            return this;
        }

        public Builder setTokenProviderDuty2(Supplier<String> tokenProviderDuty2) {
            this.tokenProviderDuty2 = tokenProviderDuty2;
            return this;
        }

        public Builder setConnectTimeout(Duration connectTimeout) {
            this.connectTimeout = connectTimeout;
            return this;
        }

        public Builder setRequestTimeout(Duration requestTimeout) {
            this.requestTimeout = requestTimeout;
            return this;
        }

        public Builder setHandlerExecutor(Executor handlerExecutor) {
            this.handlerExecutor = handlerExecutor;
            return this;
        }

        public AbcClientOptions build() {
            return new AbcClientOptions(this);
        }
    }
}
