package ru.yandex.abc;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.time.Duration;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.function.Supplier;

import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Charsets;

import ru.yandex.abc.dto.AbcMember;
import ru.yandex.abc.dto.AbcService;
import ru.yandex.abc.dto.AbcServiceDuty;
import ru.yandex.abc.dto.AbcServiceDuty2;
import ru.yandex.abc.dto.Page;
import ru.yandex.abc.dto.PageDuty2;


/**
 * @author snoop
 * @author Sergey Polovko
 */
public class HttpAbcClient implements AbcClient {

    private final String url;
    private final Duration requestTimeout;
    private final String oauthToken;
    private final HttpClient httpClient;
    private final Supplier<String> tokenProviderDuty2;
    private final Supplier<String> tokenHeaderProviderDuty2;
    private final ObjectMapper mapper = new ObjectMapper();

    public HttpAbcClient(AbcClientOptions opts) {
        this.url = opts.getUrl();
        this.requestTimeout = opts.getRequestTimeout();
        this.oauthToken = opts.getOauthToken();
        this.tokenProviderDuty2 = opts.getTokenProviderDuty2();
        this.tokenHeaderProviderDuty2 = opts.getTokenHeaderProviderDuty2();

        this.httpClient = HttpClient.newBuilder()
                .version(HttpClient.Version.HTTP_1_1)
                .followRedirects(HttpClient.Redirect.NEVER)
                .connectTimeout(opts.getConnectTimeout())
                .executor(opts.getHandlerExecutor())
                .build();
    }

    @Override
    public CompletableFuture<Optional<AbcService>> getService(String slug) {
        try {
            URI endpoint = URI.create(url + "/api/v4/services/?fields=slug,id,state&slug=" + slug);
            var request = HttpRequest.newBuilder(endpoint)
                    .header("Authorization", "OAuth " + oauthToken)
                    .header("Accept", "application/json")
                    .timeout(requestTimeout)
                    .build();

            return httpClient.sendAsync(request, HttpResponse.BodyHandlers.ofString(Charsets.UTF_8))
                    .thenApply(response -> {
                        if (response.statusCode() == 200) {
                            return parseServiceResponse(response.body());
                        }

                        String message = "cannot get ABC service from " + endpoint +
                                ", code: " + response.statusCode() +
                                ", response: " + response.body();
                        throw new IllegalStateException(message);
                    });
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<List<AbcMember>> getServiceMembers(String slug, String roleScopeSlug) {
        try {
            URI endpoint = URI.create(url + "/api/v4/services/members/?service__slug=" + slug + "&role__scope__slug=" + roleScopeSlug);
            var request = HttpRequest.newBuilder(endpoint)
                    .header("Authorization", "OAuth " + oauthToken)
                    .header("Accept", "application/json")
                    .timeout(requestTimeout)
                    .build();

            return httpClient.sendAsync(request, HttpResponse.BodyHandlers.ofString(Charsets.UTF_8))
                    .thenApply(response -> {
                        if (response.statusCode() == 200) {
                            return parseServiceMemberResponse(response.body());
                        }

                        String message = "cannot get ABC service members from " + endpoint +
                                ", code: " + response.statusCode() +
                                ", response: " + response.body();
                        throw new IllegalStateException(message);
                    });
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<List<AbcServiceDuty>> getServiceDuty(String slug) {
        try {
            URI endpoint = URI.create(url + "/api/v4/duty/schedules/?fields=slug,id&service__slug=" + slug);
            var request = HttpRequest.newBuilder(endpoint)
                    .header("Authorization", "OAuth " + oauthToken)
                    .header("Accept", "application/json")
                    .timeout(requestTimeout)
                    .build();

            return httpClient.sendAsync(request, HttpResponse.BodyHandlers.ofString(Charsets.UTF_8))
                    .thenApply(response -> {
                        if (response.statusCode() == 200) {
                            return parseServiceDutyResponse(response.body());
                        }

                        String message = "cannot get ABC service duty from " + endpoint +
                                ", code: " + response.statusCode() +
                                ", response: " + response.body();
                        throw new IllegalStateException(message);
                    });
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    @Override
    public CompletableFuture<Optional<AbcServiceDuty2>> getDutyVersion2(String slug) {
        try {
            URI endpoint = URI.create(url + "/api/watcher/v1/schedule/?filter=slug=" + slug);
            var request = HttpRequest.newBuilder(endpoint)
                    .header(tokenHeaderProviderDuty2.get(), tokenProviderDuty2.get())
                    .header("Accept", "application/json")
                    .timeout(requestTimeout)
                    .build();

            return httpClient.sendAsync(request, HttpResponse.BodyHandlers.ofString(Charsets.UTF_8))
                    .thenApply(response -> {
                        if (response.statusCode() == 200) {
                            return parseServiceDuty2Response(response.body());
                        }

                        String message = "cannot get ABC service duty version 2 from " + endpoint +
                                ", code: " + response.statusCode() +
                                ", response: " + response.body();
                        throw new IllegalStateException(message);
                    });
        } catch (Throwable t) {
            return CompletableFuture.failedFuture(t);
        }
    }

    private List<AbcMember> parseServiceMemberResponse(String body) {
        JavaType type = mapper.getTypeFactory().constructParametricType(Page.class, AbcMember.class);
        try {
            Page<AbcMember> abcServices = mapper.readValue(body, type);
            if (abcServices.getResults().isEmpty()) {
                return List.of();
            }
            return abcServices.getResults();
        } catch (IOException e) {
            throw new UncheckedIOException("cannot parse JSON: " + body, e);
        }
    }

    private Optional<AbcServiceDuty2> parseServiceDuty2Response(String body) {
        JavaType type = mapper.getTypeFactory().constructParametricType(PageDuty2.class, AbcServiceDuty2.class);
        try {
            PageDuty2<AbcServiceDuty2> abcServices = mapper.readValue(body, type);
            if (abcServices.getResults().isEmpty()) {
                return Optional.empty();
            }
            return Optional.of(abcServices.getResults().get(0));
        } catch (IOException e) {
            throw new UncheckedIOException("cannot parse JSON: " + body, e);
        }
    }

    private List<AbcServiceDuty> parseServiceDutyResponse(String body) {
        JavaType type = mapper.getTypeFactory().constructParametricType(Page.class, AbcServiceDuty.class);
        try {
            Page<AbcServiceDuty> abcServices = mapper.readValue(body, type);
            if (abcServices.getResults().isEmpty()) {
                return List.of();
            }
            return abcServices.getResults();
        } catch (IOException e) {
            throw new UncheckedIOException("cannot parse JSON: " + body, e);
        }
    }

    private Optional<AbcService> parseServiceResponse(String body) {
        JavaType type = mapper.getTypeFactory().constructParametricType(Page.class, AbcService.class);
        try {
            Page<AbcService> abcServices = mapper.readValue(body, type);
            if (abcServices.getResults().isEmpty()) {
                return Optional.empty();
            }
            return Optional.of(abcServices.getResults().get(0));
        } catch (IOException e) {
            throw new UncheckedIOException("cannot parse JSON: " + body, e);
        }
    }
}
