package ru.yandex.ambry;

import java.time.Clock;
import java.time.Instant;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.ambry.dto.LastUpdatedResponse;
import ru.yandex.ambry.dto.YasmAlertDto;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class AmbryClientStub implements AmbryClient {
    private final ConcurrentHashMap<String, YasmAlertDto> alerts = new ConcurrentHashMap<>();
    private volatile Instant lastUpdated;
    private final Clock clock;

    public AmbryClientStub(Clock clock) {
        this.clock = clock;
        this.lastUpdated = clock.instant();
    }

    public void add(YasmAlertDto alert) {
        alerts.put(alert.name, alert);
        lastUpdated = clock.instant();
    }

    public void remove(String alertName) {
        alerts.remove(alertName);
        lastUpdated = clock.instant();
    }

    @Override
    public CompletableFuture<List<YasmAlertDto>> list(TagFormat tagFormat, int limit, int offset) {
        return CompletableFuture.supplyAsync(() -> alerts.values().stream()
                .skip(offset)
                .limit(limit < 0 ? Long.MAX_VALUE : limit)
                .collect(Collectors.toList()));
    }

    @Override
    public CompletableFuture<LastUpdatedResponse> lastUpdatedTotal() {
        return CompletableFuture.supplyAsync(() -> {
            LastUpdatedResponse response = new LastUpdatedResponse();
            response.response = new LastUpdatedResponse.Response();
            synchronized (alerts) {
                response.response.total = alerts.size();
                response.response.lastUpdated = lastUpdated;
            }
            return response;
        });
    }
}
