package ru.yandex.ambry;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.time.Duration;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.databind.ObjectMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.ambry.dto.AbstractResponse;
import ru.yandex.ambry.dto.LastUpdatedResponse;
import ru.yandex.ambry.dto.ListResponse;
import ru.yandex.ambry.dto.YasmAlertDto;
import ru.yandex.misc.io.http.HttpStatus;
import ru.yandex.monlib.metrics.registry.MetricRegistry;
import ru.yandex.solomon.selfmon.failsafe.CircuitBreaker;
import ru.yandex.solomon.selfmon.failsafe.ExpMovingAverageCircuitBreaker;
import ru.yandex.solomon.selfmon.http.HttpClientMetrics;

import static java.util.concurrent.CompletableFuture.failedFuture;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class HttpAmbryClient implements AmbryClient {
    private final Logger logger = LoggerFactory.getLogger(HttpAmbryClient.class);

    private final String host;
    private final HttpClient httpClient;
    private final HttpClientMetrics metrics;
    private final ObjectMapper mapper = new ObjectMapper();
    private final CircuitBreaker circuitBreaker =
            new ExpMovingAverageCircuitBreaker(0.4, TimeUnit.MILLISECONDS.toMillis(30));

    private final Duration requestTimeout;

    public HttpAmbryClient(String host, MetricRegistry registry, Executor executor) {
        this(host, registry, executor, Duration.ofSeconds(15), Duration.ofMinutes(1));
    }

    public HttpAmbryClient(
            String host,
            MetricRegistry registry,
            Executor executor,
            Duration connectTimeout,
            Duration requestTimeout)
    {
        this.httpClient = HttpClient.newBuilder()
                .connectTimeout(connectTimeout)
                .executor(executor)
                .followRedirects(HttpClient.Redirect.NEVER)
                .version(HttpClient.Version.HTTP_1_1)
                .build();
        this.host = host;
        this.metrics = new HttpClientMetrics("ambry", registry);
        this.requestTimeout = requestTimeout;
    }

    @Override
    public CompletableFuture<List<YasmAlertDto>> list(TagFormat tagFormat, int limit, int offset) {
        if (!circuitBreaker.attemptExecution()) {
            return failedFuture(new RuntimeException("CircuitBreaker is open for " + host));
        }

        var endpoint = "/alerts/list";
        var urlStr = host + endpoint + "?tag_format=" + tagFormat.name().toLowerCase();
        if (limit >= 0) {
            urlStr = urlStr + "&limit=" + limit;
        }
        if (offset > 0) {
            urlStr = urlStr + "&offset=" + offset;
        }
        var url = URI.create(urlStr);

        var request = HttpRequest.newBuilder(url).GET();
        return call(endpoint, request, ListResponse.class)
                .thenApply(listResponse -> listResponse.response.result);
    }

    @Override
    public CompletableFuture<LastUpdatedResponse> lastUpdatedTotal() {
        if (!circuitBreaker.attemptExecution()) {
            return failedFuture(new RuntimeException("CircuitBreaker is open for " + host));
        }

        var endpoint = "/alerts/last_updated_total";
        var url = URI.create(host + endpoint);

        var request = HttpRequest.newBuilder(url).GET();
        return call(endpoint, request, LastUpdatedResponse.class);
    }

    private <T extends AbstractResponse> CompletableFuture<T> call(
            String endpointName,
            HttpRequest.Builder requestBuilder,
            Class<T> clazz)
    {
        var request = requestBuilder
                .header("X-Request-Id", UUID.randomUUID().toString())
                .header("Content-Type", "application/json")
                .header("Accept", "application/json")
                .timeout(requestTimeout)
                .build();
        var endpoint = metrics.endpoint(endpointName);
        try {
            CompletableFuture<HttpResponse<byte[]>> apiCall = endpoint.callMetrics.wrapFuture(() ->
                    httpClient.sendAsync(request, HttpResponse.BodyHandlers.ofByteArray()));

            return apiCall.whenComplete((response, exception) -> {
                        if (exception != null) {
                            circuitBreaker.markFailure();
                            logger.error("Call to " + endpointName + " failed", exception);
                        }

                        if (response != null) {
                            int status = response.statusCode();
                            endpoint.incStatus(status);
                            if (HttpStatus.is5xx(status)) {
                                circuitBreaker.markFailure();
                            } else {
                                circuitBreaker.markSuccess();
                            }
                        }
                    })
                    .thenApply(jsonResponse -> {
                        try {
                            return mapper.readValue(jsonResponse.body(), clazz);
                        } catch (IOException e) {
                            throw new UncheckedIOException("Failed to deserialize response in " + endpointName, e);
                        }
                    })
                    .thenApply(response -> validateResponse(endpointName, response));
        } catch (Exception e) {
            circuitBreaker.markFailure();
            logger.error("Unhandled exception in " + endpointName, e);
            return failedFuture(e);
        }
    }

    private <T extends AbstractResponse> T validateResponse(String endpoint, T response) {
        if (!"ok".equals(response.status)) {
            throw new AmbryException(endpoint, response.status);
        }
        return response;
    }
}
