package ru.yandex.ambry.dto;

import java.io.IOException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Spliterators;
import java.util.stream.Collectors;
import java.util.stream.StreamSupport;

import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.JsonDeserializer;
import com.fasterxml.jackson.databind.JsonNode;
import com.google.common.base.Splitter;


/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class TagsField {
    public static class Deserializer extends JsonDeserializer<Map<String, List<String>>> {

        @Override
        public Map<String, List<String>> deserialize(
                JsonParser p,
                DeserializationContext ctxt) throws IOException, JsonProcessingException
        {
            JsonNode node = p.readValueAsTree();
            if (node.isObject()) {
                return parseTagsObject(node);
            }
            if (node.isTextual()) {
                return parseJoinedTags(node.textValue());
            }
            throw ctxt.mappingException("Unknown format for tags field");
        }
    }

    private static Map<String, List<String>> parseTagsObject(JsonNode object) {
        Map<String, List<String>> result = new HashMap<>(object.size());
        object.fields()
                .forEachRemaining(entry -> result.put(entry.getKey(),
                        StreamSupport.stream(Spliterators.spliteratorUnknownSize(entry.getValue().iterator(), 0), false)
                                .map(JsonNode::textValue)
                                .collect(Collectors.toUnmodifiableList())));
        return result;
    }

    private static List<String> parseValue(String commaJoined) {
        return StreamSupport.stream(Splitter.on(',').split(commaJoined).spliterator(), false)
                .map(String::trim)
                .collect(Collectors.toList());
    }

    public static Map<String, List<String>> parseJoinedTags(String textValue) {
        return Splitter.on(';').withKeyValueSeparator('=').split(textValue).entrySet().stream()
                .collect(Collectors.toMap(entry -> entry.getKey().trim(), e -> parseValue(e.getValue())));
    }
}
