package ru.yandex.ambry.http;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.time.Instant;
import java.util.List;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.io.Resources;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.mockserver.junit.MockServerRule;
import org.mockserver.model.HttpRequest;
import org.mockserver.model.HttpResponse;

import ru.yandex.ambry.AmbryClient;
import ru.yandex.ambry.AmbryException;
import ru.yandex.ambry.HttpAmbryClient;
import ru.yandex.ambry.dto.LastUpdatedResponse;
import ru.yandex.ambry.dto.YasmAlertDto;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

import static ru.yandex.misc.concurrent.CompletableFutures.join;
import static ru.yandex.misc.concurrent.CompletableFutures.unwrapCompletionException;

/**
 * @author Ivan Tsybulin
 */
@ParametersAreNonnullByDefault
public class HttpAmbryClientTest {
    private AmbryClient client;
    private Executor executor;
    private MetricRegistry registry;
    @Rule
    public final MockServerRule mockServerRule = new MockServerRule(this);

    @Before
    public void setUp() {
        executor = Executors.newFixedThreadPool(2);
        registry = new MetricRegistry();
        client = new HttpAmbryClient("http://localhost:" + mockServerRule.getPort(), registry, executor);
    }

    @Test
    public void lastUpdated() {
        mockServerRule.getClient()
                .when(HttpRequest.request()
                        .withMethod("GET")
                        .withPath("/alerts/last_updated_total")
                        .withHeader("Content-Type", "application/json")
                        .withHeader("Accept", "application/json"))
                .respond(HttpResponse.response(load("lastUpdated.json")));

        LastUpdatedResponse response = join(client.lastUpdatedTotal());
        Assert.assertEquals(Instant.parse("2020-01-01T00:00:00Z"), response.response.lastUpdated);
        Assert.assertEquals(100500, response.response.total.intValue());
    }

    @Test(expected = AmbryException.class)
    public void lastUpdatedFailed() throws Throwable {
        mockServerRule.getClient()
                .when(HttpRequest.request()
                        .withMethod("GET")
                        .withPath("/alerts/last_updated_total")
                        .withHeader("Content-Type", "application/json")
                        .withHeader("Accept", "application/json"))
                .respond(HttpResponse.response(load("lastUpdatedFailed.json")));

        try {
            join(client.lastUpdatedTotal());
        } catch (Exception e) {
            throw unwrapCompletionException(e);
        }
    }

    @Test
    public void listEmpty() {
        mockServerRule.getClient()
                .when(HttpRequest.request()
                        .withMethod("GET")
                        .withPath("/alerts/list")
                        .withQueryStringParameter("tag_format", "dynamic")
                        .withHeader("Content-Type", "application/json")
                        .withHeader("Accept", "application/json"))
                .respond(HttpResponse.response(load("listEmpty.json")));

        List<YasmAlertDto> response = join(client.list(AmbryClient.TagFormat.DYNAMIC));
        Assert.assertEquals(0, response.size());
    }

    @Test
    public void list() {
        mockServerRule.getClient()
                .when(HttpRequest.request()
                        .withMethod("GET")
                        .withPath("/alerts/list")
                        .withQueryStringParameter("tag_format", "dynamic")
                        .withHeader("Content-Type", "application/json")
                        .withHeader("Accept", "application/json"))
                .respond(HttpResponse.response(load("list.json")));

        List<YasmAlertDto> response = join(client.list(AmbryClient.TagFormat.DYNAMIC));
        Assert.assertEquals(10, response.size());
    }

    @Test
    public void listDict() {
        mockServerRule.getClient()
                .when(HttpRequest.request()
                        .withMethod("GET")
                        .withPath("/alerts/list")
                        .withQueryStringParameter("tag_format", "dict")
                        .withHeader("Content-Type", "application/json")
                        .withHeader("Accept", "application/json"))
                .respond(HttpResponse.response(load("listDict.json")));

        List<YasmAlertDto> response = join(client.list(AmbryClient.TagFormat.DICT));
        Assert.assertEquals(10, response.size());
    }

    private String load(String res) {
        try {
            URL url = HttpAmbryClientTest.class.getResource(res);
            return Resources.toString(url, StandardCharsets.UTF_8);
        } catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }
}
