package ru.yandex.blackbox.http;

import java.util.Arrays;
import java.util.concurrent.CompletableFuture;

import javax.annotation.WillNotClose;

import com.google.common.base.Joiner;
import com.google.common.base.Strings;
import org.asynchttpclient.AsyncHttpClient;
import org.asynchttpclient.RequestBuilder;
import org.asynchttpclient.Response;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.blackbox.BlackboxClient;
import ru.yandex.blackbox.BlackboxClientOptions;
import ru.yandex.blackbox.NeedResetCookieException;
import ru.yandex.blackbox.UserInfo;
import ru.yandex.blackbox.http.HttpBlackboxClientMetrics.Endpoint;

/**
 * @author Vladimir Gordiychuk
 * @author Sergey Polovko
 */
public class HttpBlackboxClient implements BlackboxClient {
    private static final Logger logger = LoggerFactory.getLogger(HttpBlackboxClient.class);

    private final String url;
    private final String host;
    @WillNotClose
    private final AsyncHttpClient client;
    private final HttpBlackboxClientMetrics metrics;

    public HttpBlackboxClient(@WillNotClose AsyncHttpClient client, BlackboxClientOptions opts) {
        this.url = opts.getUrl();
        this.host = opts.getHost();
        this.client = client;
        this.metrics = new HttpBlackboxClientMetrics(opts.getRegistry());
    }

    @Override
    public CompletableFuture<UserInfo> userInfo(String login, String userIp) {
        Endpoint endpoint = metrics.getEndpoint("userinfo.login");
        endpoint.started();
        long startNanos = System.nanoTime();

        var request = new RequestBuilder()
                .setUrl(url + "/blackbox")
                .addQueryParam("method", "userinfo")
                .addQueryParam("login", login)
                .addQueryParam("userip", userIp)
                .addQueryParam("format", "json")
                .setFollowRedirect(false)
                .build();

        var future = client.executeRequest(request)
                .toCompletableFuture()
                .thenApply(response -> {
                    if (logger.isDebugEnabled()) {
                        logger.debug(
                                "blackbox response for userInfo (login={}, userIp={}): {} {}",
                                login, userIp, response.getStatusCode(), response.getResponseBody());
                    }
                    checkIsHttpOk(response);
                    return parseUserInfo(response);
                });

        return endpoint.subscribe(startNanos, future);
    }

    @Override
    public CompletableFuture<UserInfo> userInfo(long uid, String userIp) {
        Endpoint endpoint = metrics.getEndpoint("userinfo.uid");
        endpoint.started();
        long startNanos = System.nanoTime();

        var request = new RequestBuilder()
                .setUrl(url + "/blackbox")
                .addQueryParam("method", "userinfo")
                .addQueryParam("uid", String.valueOf(uid))
                .addQueryParam("userip", userIp)
                .addQueryParam("format", "json")
                .setFollowRedirect(false)
                .build();

        var future = client.executeRequest(request)
                .toCompletableFuture()
                .thenApply(response -> {
                    if (logger.isDebugEnabled()) {
                        logger.debug(
                                "blackbox response for userInfo (uid={}, userIp={}): {} {}",
                                uid, userIp, response.getStatusCode(), response.getResponseBody());
                    }
                    checkIsHttpOk(response);
                    return parseUserInfo(response);
                });

        return endpoint.subscribe(startNanos, future);
    }

    @Override
    public CompletableFuture<UserInfo> sessionId(String sessionId, String sslSessionId, String userIp) {
        Endpoint endpoint = metrics.getEndpoint("sessionid");
        endpoint.started();
        long startNanos = System.nanoTime();

        var request = new RequestBuilder()
                .setUrl(url + "/blackbox")
                .addQueryParam("method", "sessionid")
                .addQueryParam("sessionid", sessionId)
                .addQueryParam("userip", userIp)
                .addQueryParam("host", host)
                .addQueryParam("format", "json")
                .setFollowRedirect(false);

        if (!sslSessionId.isEmpty()) {
            request = request.addQueryParam("sslsessionid", sslSessionId);
        }

        var future = client.executeRequest(request.build())
                .toCompletableFuture()
                .thenApply(response -> {
                    if (logger.isDebugEnabled()) {
                        logger.debug(
                                "blackbox response for sessionId (sessionId={}, sslSessionId={}, userIp={}): {} {}",
                                Strings.isNullOrEmpty(sessionId) ? "empty" : "***",    // do not log real session id
                                Strings.isNullOrEmpty(sslSessionId) ? "empty" : "***", // do not log real session id
                                userIp,
                                response.getStatusCode(),
                                response.getResponseBody());
                    }
                    checkIsHttpOk(response);
                    return parseSessionId(response);
                });

        return endpoint.subscribe(startNanos, future);
    }

    @Override
    public CompletableFuture<UserInfo> oauth(String token, String[] scopes, String userIp) {
        Endpoint endpoint = metrics.getEndpoint("oauth");
        endpoint.started();
        long startNanos = System.nanoTime();

        var request = new RequestBuilder()
                .setUrl(url + "/blackbox")
                .addQueryParam("method", "oauth")
                .addQueryParam("oauth_token", token)
                .addQueryParam("userip", userIp)
                .addQueryParam("format", "json")
                .setFollowRedirect(false);

        if (scopes.length != 0) {
            request.addQueryParam("scopes", Joiner.on(',').join(scopes));
        }

        var future = client.executeRequest(request)
                .toCompletableFuture()
                .thenApply(response -> {
                    if (logger.isDebugEnabled()) {
                        logger.debug(
                                "blackbox response for oauth (token={}, scopes={}, userIp={}): {} {}",
                                Strings.isNullOrEmpty(token) ? "empty" : "***",    // do not log real token
                                Arrays.toString(scopes),
                                userIp,
                                response.getStatusCode(),
                                response.getResponseBody());
                    }
                    checkIsHttpOk(response);
                    return parseOauth(response);
                });

        return endpoint.subscribe(startNanos, future);
    }

    static UserInfo parseUserInfo(Response response) {
        var userInfoResponse = ResponseParser.parseUserInfo(response.getResponseBody());
        checkIsOk(response, userInfoResponse.error);

        if (userInfoResponse.users.length == 0) {
            return null;
        }

        var userInfoDto = userInfoResponse.users[0];

        if (userInfoDto.login == null) {
            logger.error("Login is required in blackbox response, but: {}", response.getResponseBody());
            throw new IllegalStateException("Invalid blackbox response: login is required");
        }

        return new UserInfo(userInfoDto.id, userInfoDto.login);
    }

    static UserInfo parseSessionId(Response response) {
        var sessionIdResponse = ResponseParser.parseSessionId(response.getResponseBody());
        checkIsOk(response, sessionIdResponse.error);

        String status = sessionIdResponse.status.value;
        if ("NEED_RESET".equals(status)) {
            throw new NeedResetCookieException("need to reset cookie");
        }

        if (!"VALID".equals(status)) {
            throw new NeedResetCookieException("expected 'VALID' or 'NEED_RESET' status, but got " + status);
        }

        return new UserInfo(sessionIdResponse.id, sessionIdResponse.login);
    }

    UserInfo parseOauth(Response response) {
        var oauthResponse = ResponseParser.parseOauth(response.getResponseBody());
        checkIsOk(response, oauthResponse.error);

        String status = oauthResponse.status.value;
        if (!"VALID".equals(status)) {
            throw new NeedResetCookieException("expected 'VALID' status, but got " + status);
        }

        return new UserInfo(oauthResponse.uid.value, oauthResponse.login);
    }

    private static void checkIsOk(Response response, String error) {
        if (!"".equals(error) && !"OK".equals(error)) {
            logger.debug("Blackbox error: {}", response.getResponseBody());
            throw new IllegalStateException(error);
        }
    }

    private static void checkIsHttpOk(Response response) {
        if (response.getStatusCode() != 200) {
            logger.error("Blackbox invalid status {}: {}", response.getStatusCode(), response.getResponseBody());
            String message = "Blackbox response not 200: " + response.getStatusCode() + ' ' + response.getResponseBody();
            throw new IllegalStateException(message);
        }
    }
}
