package ru.yandex.blackbox.http;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.TimeUnit;

import ru.yandex.monlib.metrics.histogram.Histograms;
import ru.yandex.monlib.metrics.primitives.Histogram;
import ru.yandex.monlib.metrics.primitives.LazyGaugeInt64;
import ru.yandex.monlib.metrics.primitives.Rate;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

/**
 * @author Vladimir Gordiychuk
 */
public class HttpBlackboxClientMetrics {

    private final MetricRegistry registry;
    private final ConcurrentMap<String, Endpoint> metricsByEndpoint = new ConcurrentHashMap<>();

    public HttpBlackboxClientMetrics(MetricRegistry registry) {
        this.registry = registry;
    }

    public Endpoint getEndpoint(String endpoint) {
        return metricsByEndpoint.computeIfAbsent(endpoint, name -> new Endpoint(registry, name));
    }

    public static class Endpoint {
        private final Rate started;
        private final Rate completed;
        private final Rate failed;
        private final LazyGaugeInt64 inFlight;
        private final Histogram elapsedTimeMillis;

        public Endpoint(MetricRegistry registry, String endpoint) {
            this(registry.subRegistry("endpoint", endpoint));
        }

        public Endpoint(MetricRegistry registry) {
            started = registry.rate("blackbox.client.request.started");
            completed = registry.rate("blackbox.client.request.completed");
            failed = registry.rate("blackbox.client.request.failed");
            inFlight = registry.lazyGaugeInt64("blackbox.client.request.inFlight", () -> started.get() - completed.get() - failed.get());
            elapsedTimeMillis = registry.histogramRate("blackbox.client.request.elapsedTimeMillis", Histograms.exponential(13, 2, 16));
        }

        public void started() {
            this.started.inc();
        }

        public void failed() {
            this.failed.inc();
        }

        public void complted(long elapsedTimeNanos) {
            completed.inc();
            elapsedTimeMillis.record(TimeUnit.NANOSECONDS.toMillis(elapsedTimeNanos));
        }

        public <T> CompletableFuture<T> subscribe(long startNanos, CompletableFuture<T> future) {
            return future.whenComplete((result, e) -> {
                if (e == null) {
                    complted(System.nanoTime() - startNanos);
                } else {
                    failed();
                }
            });
        }
    }
}
