package ru.yandex.cloud.auth.token;

import java.time.Duration;
import java.time.Instant;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.cloud.token.IamToken;
import ru.yandex.cloud.token.IamTokenClient;
import ru.yandex.cloud.token.Jwt;

import static java.util.Objects.requireNonNull;

/**
 * @author Sergey Polovko
 */
class IamTokenProvider implements TokenProvider {
    private static final Logger logger = LoggerFactory.getLogger(IamTokenProvider.class);

    private final IamTokenClient tokenClient;
    private final Jwt.Builder jwtBuilder;
    private final ScheduledExecutorService scheduler;

    private volatile IamToken token;
    private volatile CompletableFuture<IamToken> tokenFuture;

    IamTokenProvider(IamTokenClient tokenClient, Jwt.Builder jwtBuilder, ScheduledExecutorService scheduler) {
        this.tokenClient = tokenClient;
        this.jwtBuilder = jwtBuilder.clone();
        this.scheduler = requireNonNull(scheduler);
        this.tokenFuture = updateToken();
    }

    @Override
    public String getToken() {
        IamToken token = this.token;
        if (token != null) {
            return token.getValue();
        }
        return tokenFuture.join().getValue();
    }

    private CompletableFuture<IamToken> updateToken() {
        return tokenClient.createByJwt(jwtBuilder.build())
                .whenComplete((newToken, throwable) -> {
                    if (throwable != null) {
                        logger.warn("cannot fetch IAM token", throwable);
                    } else {
                        token = newToken;
                    }

                    Duration delay = Duration.ofSeconds(3);
                    if (newToken != null) {
                        var expiration = Duration.between(Instant.now(), newToken.getExpiresAt());
                        if (!expiration.isNegative()) {
                            delay = expiration.dividedBy(2);
                        }
                    }

                    scheduler.schedule(() -> {
                        tokenFuture = updateToken();
                    }, delay.toMillis(), TimeUnit.MILLISECONDS);
                });
    }
}
