package ru.yandex.cloud.client.compute;

import java.time.Duration;
import java.util.concurrent.Executor;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.WillNotClose;

import ru.yandex.cloud.auth.token.TokenProvider;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

import static java.util.Objects.requireNonNull;


/**
 * @author Sergey Polovko
 */
@ParametersAreNonnullByDefault
public class ComputeClientOptions {

    private static final String HOST_PROD = "compute.private-api.cloud.yandex.net";
    private static final String HOST_PREPROD = "compute.private-api.cloud-preprod.yandex.net";

    private static final int DEFAULT_HTTP_PORT = 9000;

    private final String host;
    private final int port;
    @Nullable
    private Executor handlerExecutor;
    private Duration connectTimeout = Duration.ofSeconds(5);
    private Duration requestTimeout = Duration.ofSeconds(10);
    private TokenProvider tokenProvider = TokenProvider.of("");
    private String folderId = "";
    private MetricRegistry registry = MetricRegistry.root();

    private ComputeClientOptions(String host, int port) {
        this.host = host;
        this.port = port;
    }

    public String getHost() {
        return host;
    }

    public int getPort() {
        return port;
    }

    @Nullable
    public Executor getHandlerExecutor() {
        return handlerExecutor;
    }

    public Duration getConnectTimeout() {
        return connectTimeout;
    }

    public Duration getRequestTimeout() {
        return requestTimeout;
    }

    public TokenProvider getTokenProvider() {
        return tokenProvider;
    }

    public String getFolderId() {
        return folderId;
    }

    public MetricRegistry getMetricRegistry() {
        return registry;
    }

    public static ComputeClientOptions forAddress(String host, int port) {
        return new ComputeClientOptions(requireNonNull(host, "host"), port);
    }

    public static ComputeClientOptions forProd() {
        return forAddress(HOST_PROD, DEFAULT_HTTP_PORT);
    }

    public static ComputeClientOptions forPreprod() {
        return forAddress(HOST_PREPROD, DEFAULT_HTTP_PORT);
    }

    public ComputeClientOptions withHandlerExecutor(@WillNotClose Executor executor) {
        this.handlerExecutor = requireNonNull(executor);
        return this;
    }

    public ComputeClientOptions withConnectTimeout(Duration timeout) {
        this.connectTimeout = requireNonNull(timeout);
        return this;
    }

    public ComputeClientOptions withRequestTimeout(Duration timeout) {
        this.requestTimeout = requireNonNull(timeout);
        return this;
    }

    public ComputeClientOptions withTokenProvider(TokenProvider tokenProvider) {
        this.tokenProvider = requireNonNull(tokenProvider);
        return this;
    }

    public ComputeClientOptions withFolderId(String folderId) {
        this.folderId = requireNonNull(folderId);
        return this;
    }

    public ComputeClientOptions withMetricsRegistry(MetricRegistry registry) {
        this.registry = requireNonNull(registry);
        return this;
    }
}
