package ru.yandex.cloud.client.compute.http;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.util.List;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import com.fasterxml.jackson.databind.ObjectMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.cloud.auth.Headers;
import ru.yandex.cloud.auth.token.TokenProvider;
import ru.yandex.cloud.client.compute.ComputeClient;
import ru.yandex.cloud.client.compute.ComputeClientOptions;
import ru.yandex.cloud.client.compute.Resource;
import ru.yandex.cloud.client.compute.ResourcesPage;
import ru.yandex.solomon.selfmon.http.HttpClientMetrics;


/**
 * @author Sergey Polovko
 */
public class HttpComputeClient implements ComputeClient {
    private static final Logger logger = LoggerFactory.getLogger(HttpComputeClient.class);

    private final ObjectMapper objectMapper = new ObjectMapper();
    private final String apiEndpoint;
    private final TokenProvider tokenProvider;
    private final String folderId;
    private final Duration requestTimeout;
    private final HttpClientMetrics metrics;
    private final HttpClient httpClient;

    public HttpComputeClient(ComputeClientOptions opts) {
        var scheme = (opts.getPort() == 443) ? "https://" : "http://";
        this.apiEndpoint = scheme + opts.getHost() + ':' + opts.getPort() + "/external/v1";
        this.tokenProvider = opts.getTokenProvider();
        this.folderId = opts.getFolderId();
        this.requestTimeout = opts.getRequestTimeout();
        this.metrics = new HttpClientMetrics("compute", opts.getMetricRegistry());
        this.httpClient = HttpClient.newBuilder()
            .connectTimeout(opts.getConnectTimeout())
            .executor(opts.getHandlerExecutor())
            .followRedirects(HttpClient.Redirect.NEVER)
            .version(HttpClient.Version.HTTP_1_1)
            .build();
    }

    @Override
    public CompletableFuture<Set<String>> types() {
        String uri = apiEndpoint + "/solomon/types";
        return fetchJson(uri, Dto.Types.class)
            .thenApply(r -> r.types);
    }

    @Override
    public CompletableFuture<ResourcesPage> resources(String resourceType, @Nullable String pageToken) {
        String uri = apiEndpoint + "/solomon/resources" +
            "?folderId=" + folderId +
            "&pageSize=500" +
            "&resourceType=" + resourceType;
        if (pageToken != null && !pageToken.isEmpty()) {
            uri += "&pageToken=" + pageToken;
        }
        return fetchJson(uri, Dto.ResourcesPage.class)
            .thenApply(page -> {
                List<Resource> resources = page.resources.stream()
                    .map(r -> new Resource(r.cloudId, r.folderId, r.id, r.name == null ? "" : r.name, resourceType))
                    .collect(Collectors.toList());
                return new ResourcesPage(page.pageToken, resources);
            });
    }

    private <T> CompletableFuture<T> fetchJson(String url, Class<T> entityClass) {
        String requestId = UUID.randomUUID().toString();
        var uri = URI.create(url);
        var endpoint = metrics.endpoint(uri.getPath());
        HttpRequest request = HttpRequest.newBuilder(uri)
            .setHeader("request_id", requestId)
            .setHeader(Headers.TOKEN_HEADER, tokenProvider.getToken())
            .timeout(requestTimeout)
            .build();

        if (logger.isDebugEnabled()) {
            logger.debug("requestId={} GET {}", requestId, url);
        }

        var future = httpClient.sendAsync(request, HttpResponse.BodyHandlers.ofByteArray())
            .thenApply(response -> {
                byte[] body = response.body();

                endpoint.incStatus(response.statusCode());
                if (response.statusCode() != 200) {
                    throw new IllegalStateException(String.format(
                        "requestId=%s, non OK response: %d %s",
                        requestId,
                        response.statusCode(),
                        new String(body, StandardCharsets.UTF_8)));
                }

                try {
                    return objectMapper.readValue(body, entityClass);
                } catch (IOException e) {
                    String msg = String.format(
                        "requestId=%s, cannot parse json: %s",
                        requestId,
                        new String(body, StandardCharsets.UTF_8));
                    throw new UncheckedIOException(msg, e);
                }
            });
        endpoint.callMetrics.forFuture(future);
        return future;
    }

    @Override
    public void close() {
        // nop
    }
}
