package ru.yandex.cloud.client.compute;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.charset.StandardCharsets;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

import com.google.common.io.Resources;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.mockserver.junit.MockServerRule;
import org.mockserver.model.HttpRequest;
import org.mockserver.model.HttpResponse;
import org.mockserver.model.Parameter;

import ru.yandex.cloud.auth.Headers;
import ru.yandex.cloud.auth.token.TokenProvider;
import ru.yandex.cloud.client.compute.http.HttpComputeClient;

import static com.google.common.net.MediaType.JSON_UTF_8;


/**
 * @author Sergey Polovko
 */
public class ResourcesTest {
    @Rule
    public MockServerRule mockServerRule = new MockServerRule(this);

    private ExecutorService executor;
    private ComputeClient computeClient;

    @Before
    public void setUp() {
        executor = Executors.newSingleThreadExecutor();
        computeClient = new HttpComputeClient(
            ComputeClientOptions.forAddress("localhost", mockServerRule.getPort())
                .withHandlerExecutor(executor)
                .withTokenProvider(TokenProvider.of("my-token"))
                .withFolderId("my-folder"));
    }

    @After
    public void tearDown() throws Exception {
        executor.shutdown();
        executor.awaitTermination(5, TimeUnit.SECONDS);
    }

    @Test
    public void empty() {
        mockServerRule.getClient()
            .when(httpRequest(Parameter.param("resourceType", "disk")))
            .respond(request -> httpResponse(request, "{\"resources\":[]}"));

        ResourcesPage resourcesPage = computeClient.resources("disk", null).join();
        Assert.assertNull(resourcesPage.getNextPageToken());
    }

    @Test
    public void nonEmpty() {
        // page2 (XXX: must be defined before page1)
        mockServerRule.getClient()
            .when(httpRequest(
                Parameter.param("resourceType", "instance"),
                Parameter.param("pageToken", "page2")))
            .respond(request -> httpResponse(request, readResource("resources-page2.json")));

        // page1
        mockServerRule.getClient()
            .when(httpRequest(Parameter.param("resourceType", "instance")))
            .respond(request -> httpResponse(request, readResource("resources-page1.json")));

        ResourcesPage page1 = computeClient.resources("instance", null).join();
        Assert.assertEquals(page1.getNextPageToken(), "page2");
        Assert.assertEquals(2, page1.getResources().size());
        Assert.assertEquals(
            new Resource("cloud1", "folder1", "instance1", "instance1-name", "instance"),
            page1.getResources().get(0));
        Assert.assertEquals(
            new Resource("cloud1", "folder1", "instance2", "instance2-name", "instance"),
            page1.getResources().get(1));

        ResourcesPage page2 = computeClient.resources("instance", page1.getNextPageToken()).join();
        Assert.assertEquals(
            new Resource("cloud1", "folder1", "instance3", "instance3-name", "instance"),
            page2.getResources().get(0));
    }

    private static HttpRequest httpRequest(Parameter... params) {
        return HttpRequest.request("/external/v1/solomon/resources")
            .withMethod("GET")
            .withQueryStringParameter("folderId", "my-folder")
            .withQueryStringParameters(params);
    }

    private static HttpResponse httpResponse(HttpRequest request, String body) {
        String token = request.getFirstHeader(Headers.TOKEN_HEADER);
        Assert.assertEquals("my-token", token);
        return HttpResponse.response().withBody(body, JSON_UTF_8);
    }

    private static String readResource(String name) {
        try {
            return Resources.toString(ResourcesTest.class.getResource(name), StandardCharsets.UTF_8);
        } catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }
}
