package ru.yandex.cloud.client.compute;

import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

import com.google.common.net.MediaType;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.mockserver.junit.MockServerRule;
import org.mockserver.model.HttpRequest;
import org.mockserver.model.HttpResponse;

import ru.yandex.cloud.auth.Headers;
import ru.yandex.cloud.auth.token.TokenProvider;
import ru.yandex.cloud.client.compute.http.HttpComputeClient;


/**
 * @author Sergey Polovko
 */
public class TypesTest {
    @Rule
    public MockServerRule mockServerRule = new MockServerRule(this);

    private ExecutorService executor;
    private ComputeClient computeClient;

    @Before
    public void setUp() {
        executor = Executors.newSingleThreadExecutor();
        computeClient = new HttpComputeClient(
            ComputeClientOptions.forAddress("localhost", mockServerRule.getPort())
                .withHandlerExecutor(executor)
                .withTokenProvider(TokenProvider.of("my-token")));
    }

    @After
    public void tearDown() throws Exception {
        executor.shutdown();
        executor.awaitTermination(5, TimeUnit.SECONDS);
    }

    @Test
    public void empty() {
        mockServerRule.getClient()
            .when(httpRequest())
            .respond(request -> httpResponse(request, "{\"types\":[]}"));

        Set<String> types = computeClient.types().join();
        Assert.assertTrue(types.isEmpty());
    }

    @Test
    public void nonEmpty() {
        mockServerRule.getClient()
            .when(httpRequest())
            .respond(request -> httpResponse(request, "{\"types\":[\"instance\",\"image\",\"disk\"]}"));

        Set<String> types = computeClient.types().join();
        Assert.assertEquals(Set.of("instance", "image", "disk"), types);
    }

    private static HttpRequest httpRequest() {
        return HttpRequest.request("/external/v1/solomon/types")
            .withMethod("GET");
    }

    private static HttpResponse httpResponse(HttpRequest request, String body) {
        String token = request.getFirstHeader(Headers.TOKEN_HEADER);
        Assert.assertEquals("my-token", token);
        return HttpResponse.response()
            .withBody(body, MediaType.JSON_UTF_8);
    }
}
