package ru.yandex.cloud.grpc;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.function.BiFunction;

import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.MoreExecutors;

/**
 * @author Sergey Polovko
 */
public final class Futures {
    private Futures() {}

    /**
     * Converts Guava's ListenableFuture to CompletableFuture with additional result value mapping.
     *
     * @param future      guava future
     * @param mapper      mapper function
     * @return completable future
     */
    public static <T, R> CompletableFuture<R> whenComplete(ListenableFuture<T> future, BiFunction<T, Throwable, R> mapper) {
        var promise = new CompletableFuture<R>();
        future.addListener(() -> {
            try {
                promise.complete(mapper.apply(future.get(), null));
            } catch (ExecutionException e) {
                try {
                    R apply = mapper.apply(null, e.getCause());
                    if (apply != null) {
                        promise.complete(apply);
                    } else {
                        promise.completeExceptionally(e.getCause());
                    }
                } catch (Throwable t) {
                    promise.completeExceptionally(t);
                }
            } catch (Throwable t) {
                promise.completeExceptionally(t);
            }
        }, MoreExecutors.directExecutor());
        return promise;
    }
}
