package ru.yandex.cloud.grpc;

import java.util.concurrent.Executor;

import javax.annotation.Nullable;

import com.google.protobuf.Any;
import com.google.protobuf.InvalidProtocolBufferException;
import com.google.protobuf.Message;
import com.google.rpc.Status;
import io.grpc.CallCredentials;
import io.grpc.ClientInterceptor;
import io.grpc.Metadata;
import io.grpc.protobuf.StatusProto;
import io.grpc.stub.MetadataUtils;

import ru.yandex.cloud.auth.token.TokenProvider;

/**
 * @author Sergey Polovko
 */
public final class Grpc {
    private Grpc() {}

    private static final Metadata.Key<String> AUTHORIZATION =
            Metadata.Key.of("Authorization", Metadata.ASCII_STRING_MARSHALLER);

    private static final Metadata.Key<String> REQUEST_ID =
            Metadata.Key.of("X-Request-ID", Metadata.ASCII_STRING_MARSHALLER);

    @Nullable
    public static <M extends Message> M readStatusDetails(Throwable t, Class<M> type) {
        Status status = StatusProto.fromThrowable(t);
        if (status == null) {
            return null;
        }
        for (int i = 0; i < status.getDetailsCount(); i++) {
            Any details = status.getDetails(i);
            if (details.is(type)) {
                try {
                    return details.unpack(type);
                } catch (InvalidProtocolBufferException e) {
                    throw new RuntimeException("cannot unpack status details", e);
                }
            }
        }
        return null;
    }

    public static ClientInterceptor addCallMeta(String requestId) {
        Metadata extraHeaders = new Metadata();
        extraHeaders.put(REQUEST_ID, requestId);
        return MetadataUtils.newAttachHeadersInterceptor(extraHeaders);
    }

    public static CallCredentials authCredentials(TokenProvider tokenProvider) {
        return new AuthCredentials(AUTHORIZATION, "Bearer ", tokenProvider);
    }

    public static CallCredentials authCredentials(Metadata.Key<String> header, String prefix, TokenProvider tokenProvider) {
        return new AuthCredentials(header, prefix, tokenProvider);
    }

    /**
     * AUTH CREDENTIALS
     */
    private static final class AuthCredentials extends CallCredentials {
        private final String tokenPrefix;
        private final TokenProvider tokenProvider;
        private final Metadata.Key<String> header;

        public AuthCredentials(Metadata.Key<String> header, String tokenPrefix, TokenProvider tokenProvider) {
            this.header = header;
            this.tokenPrefix = tokenPrefix;
            this.tokenProvider = tokenProvider;
        }

        @Override
        public void applyRequestMetadata(
                CallCredentials.RequestInfo requestInfo,
                Executor appExecutor,
                CallCredentials.MetadataApplier applier)
        {
            Metadata headers = new Metadata();
            headers.put(header, tokenPrefix + tokenProvider.getToken());
            applier.apply(headers);
        }

        @Override
        public void thisUsesUnstableApi() {
        }
    }
}
