package ru.yandex.cloud.client.identity;

import java.time.Duration;
import java.util.concurrent.Executor;

import javax.annotation.Nullable;
import javax.annotation.WillNotClose;

import ru.yandex.cloud.auth.token.TokenProvider;
import ru.yandex.monlib.metrics.registry.MetricRegistry;

import static java.util.Objects.requireNonNull;


/**
 * @author Sergey Polovko
 */
public class IdentityClientOptions {

    private static final String HOST_PROD = "identity.private-api.cloud.yandex.net";
    private static final String HOST_PREPROD = "identity.private-api.cloud-preprod.yandex.net";

    private static final int DEFAULT_HTTP_PORT = 2637;
    private static final int DEFAULT_HTTPS_PORT = 14336;

    private final String host;
    private final int port;
    private boolean useTls;
    @Nullable
    private Executor handlerExecutor;
    private Duration connectTimeout = Duration.ofSeconds(5);
    private Duration requestTimeout = Duration.ofSeconds(10);
    private TokenProvider tokenProvider = TokenProvider.of("");
    private MetricRegistry registry = MetricRegistry.root();

    private IdentityClientOptions(String host, int port) {
        this.host = host;
        this.port = port;
        this.useTls = port == 443;
    }
    public String getHost() {
        return host;
    }

    public int getPort() {
        return port;
    }

    public boolean isUseTls() {
        return useTls;
    }

    @Nullable
    public Executor getHandlerExecutor() {
        return handlerExecutor;
    }

    public Duration getConnectTimeout() {
        return connectTimeout;
    }

    public Duration getRequestTimeout() {
        return requestTimeout;
    }

    public TokenProvider getTokenProvider() {
        return tokenProvider;
    }

    public MetricRegistry getMetricsRegistry() {
        return registry;
    }

    public static IdentityClientOptions forAddress(String host, int port) {
        return new IdentityClientOptions(host, port);
    }

    public static IdentityClientOptions forProd() {
        return forAddress(HOST_PROD, DEFAULT_HTTPS_PORT)
                .withTls(true);
    }

    public static IdentityClientOptions forPreprod() {
        return forAddress(HOST_PREPROD, DEFAULT_HTTPS_PORT)
                .withTls(true);
    }

    public IdentityClientOptions withTls(boolean value) {
        this.useTls = value;
        return this;
    }

    public IdentityClientOptions withHandlerExecutor(@WillNotClose Executor executor) {
        this.handlerExecutor = executor;
        return this;
    }

    public IdentityClientOptions withConnectTimeout(Duration timeout) {
        this.connectTimeout = timeout;
        return this;
    }

    public IdentityClientOptions withRequestTimeout(Duration timeout) {
        this.requestTimeout = timeout;
        return this;
    }

    public IdentityClientOptions withTokenProvider(TokenProvider tokenProvider) {
        this.tokenProvider = requireNonNull(tokenProvider);
        return this;
    }

    public IdentityClientOptions withMetricsRegistry(MetricRegistry registry) {
        this.registry = requireNonNull(registry);
        return this;
    }
}
