package ru.yandex.cloud.client.identity.http;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.net.URI;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.util.List;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import com.fasterxml.jackson.databind.ObjectMapper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.cloud.auth.Headers;
import ru.yandex.cloud.auth.token.TokenProvider;
import ru.yandex.cloud.client.identity.Cloud;
import ru.yandex.cloud.client.identity.Folder;
import ru.yandex.cloud.client.identity.IdentityClient;
import ru.yandex.cloud.client.identity.IdentityClientOptions;
import ru.yandex.cloud.client.identity.User;
import ru.yandex.solomon.selfmon.http.HttpClientMetrics;


/**
 * @author Sergey Polovko
 */
@Deprecated
public class HttpIdentityClient implements IdentityClient {
    private static final Logger logger = LoggerFactory.getLogger(HttpIdentityClient.class);

    private final ObjectMapper objectMapper = new ObjectMapper();
    private final String apiEndpoint;
    private final Duration requestTimeout;
    private final TokenProvider tokenProvider;
    private final HttpClientMetrics metrics;
    private final HttpClient httpClient;

    public HttpIdentityClient(IdentityClientOptions opts) {
        var scheme = opts.isUseTls() ? "https://" : "http://";
        this.apiEndpoint = scheme + opts.getHost() + ':' + opts.getPort();
        this.requestTimeout = opts.getRequestTimeout();
        this.tokenProvider = opts.getTokenProvider();
        this.metrics = new HttpClientMetrics("identity", opts.getMetricsRegistry());
        this.httpClient = HttpClient.newBuilder()
            .connectTimeout(opts.getConnectTimeout())
            .executor(opts.getHandlerExecutor())
            .followRedirects(HttpClient.Redirect.NEVER)
            .version(HttpClient.Version.HTTP_1_1)
            .build();
    }


    public CompletableFuture<List<Cloud>> allClouds() {
        var fetcher = new PagesFetcher<Cloud, Dto.Cloud, Dto.CloudsPage>() {
            @Override
            CompletableFuture<Dto.CloudsPage> fetchPage(@Nullable String pageToken) {
                return fetchCloudsPage(pageToken);
            }

            @Override
            boolean filter(Dto.Cloud dto) {
                return "ACTIVE".equals(dto.status);
            }

            @Override
            Cloud convertDto(Dto.Cloud dto) {
                return new Cloud(dto.id, dto.name);
            }
        };
        return fetcher.getFuture();
    }

    private CompletableFuture<Dto.CloudsPage> fetchCloudsPage(@Nullable String nextPageToken) {
        String uri = apiEndpoint + "/v1/allClouds?pageSize=1000";
        if (nextPageToken != null && !nextPageToken.isEmpty()) {
            uri += "&pageToken=" + nextPageToken;
        }
        return fetchJson(uri, Dto.CloudsPage.class);
    }

    @Override
    public CompletableFuture<List<Folder>> allFolders(String cloudId) {
        var fetcher = new PagesFetcher<Folder, Dto.Folder, Dto.FoldersPage>() {
            @Override
            CompletableFuture<Dto.FoldersPage> fetchPage(@Nullable String pageToken) {
                return fetchFoldersPage(cloudId, pageToken);
            }

            @Override
            boolean filter(Dto.Folder dto) {
                return true;
            }

            @Override
            Folder convertDto(Dto.Folder dto) {
                return new Folder(dto.cloudId, dto.id, dto.name);
            }
        };
        return fetcher.getFuture();
    }

    @Override
    public CompletableFuture<Optional<Folder>> folder(String folderId) {
        return fetchFoldersPageByFolderId(folderId).thenApply(pageDto -> {
            if (pageDto.result.isEmpty()) {
                return Optional.empty();
            }

            if (pageDto.result.size() > 1) {
                throw new IllegalStateException("too many folders found by id: " + folderId);
            }

            Dto.Folder folderDto = pageDto.result.get(0);
            return Optional.of(new Folder(folderDto.cloudId, folderDto.id, folderDto.name));
        });
    }

    @Override
    public CompletableFuture<List<User>> allUsers(String cloudId) {
        String url = apiEndpoint + "/v1/clouds/" + cloudId + "/users";
        var endpoint = metrics.endpoint("/v1/clouds/:cloudId/users");
        return fetchJson(url, endpoint, Dto.Users.class)
            .thenApply(response -> response.users.stream()
                .map(user -> new User(user.id, user.login, user.firstName, user.lastName))
                .collect(Collectors.toList()));
    }

    private CompletableFuture<Dto.FoldersPage> fetchFoldersPageByFolderId(String folderId) {
        String uri = apiEndpoint + "/v1/allFolders?id=" + folderId;
        return fetchJson(uri, Dto.FoldersPage.class);
    }

    private CompletableFuture<Dto.FoldersPage> fetchFoldersPage(String cloudId, @Nullable String nextPageToken) {
        String uri = apiEndpoint + "/v1/allFolders?cloudId=" + cloudId + "&pageSize=500";
        if (nextPageToken != null && !nextPageToken.isEmpty()) {
            uri += "&pageToken=" + nextPageToken;
        }
        return fetchJson(uri, Dto.FoldersPage.class);
    }

    private <T> CompletableFuture<T> fetchJson(String url, Class<T> entityClass) {
        return fetchJson(url, metrics.endpoint(URI.create(url).getPath()), entityClass);
    }

    private <T> CompletableFuture<T> fetchJson(String url, HttpClientMetrics.Endpoint endpoint, Class<T> entityClass) {
        String requestId = UUID.randomUUID().toString();
        HttpRequest request = HttpRequest.newBuilder(URI.create(url))
            .setHeader("request_id", requestId)
            .setHeader(Headers.TOKEN_HEADER, tokenProvider.getToken())
            .timeout(requestTimeout)
            .build();

        if (logger.isDebugEnabled()) {
            logger.debug("requestId={} GET {}", requestId, url);
        }

        var future = httpClient.sendAsync(request, HttpResponse.BodyHandlers.ofByteArray())
            .thenApply(response -> {
                byte[] body = response.body();

                endpoint.incStatus(response.statusCode());
                if (response.statusCode() != 200) {
                    throw new IllegalStateException(String.format(
                        "requestId=%s, non OK response: %d %s",
                        requestId,
                        response.statusCode(),
                        new String(body, StandardCharsets.UTF_8)));
                }

                try {
                    return objectMapper.readValue(body, entityClass);
                } catch (IOException e) {
                    String msg = String.format(
                        "requestId=%s, cannot parse json: %s",
                        requestId,
                        new String(body, StandardCharsets.UTF_8));
                    throw new UncheckedIOException(msg, e);
                }
            });
        endpoint.callMetrics.forFuture(future);
        return future;
    }

    @Override
    public void close() {
        // nop
    }
}
