package ru.yandex.cloud.client.identity.http;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.charset.StandardCharsets;
import java.util.List;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

import com.google.common.io.Resources;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.mockserver.junit.MockServerRule;
import org.mockserver.model.HttpRequest;
import org.mockserver.model.HttpResponse;

import ru.yandex.cloud.client.identity.Folder;
import ru.yandex.cloud.client.identity.IdentityClient;
import ru.yandex.cloud.client.identity.IdentityClientOptions;

import static com.google.common.net.MediaType.JSON_UTF_8;


/**
 * @author Sergey Polovko
 */
public class AllFoldersTest {

    @Rule
    public MockServerRule mockServerRule = new MockServerRule(this);

    private HttpRequest template;
    private ExecutorService executor;
    private IdentityClient identityClient;

    @Before
    public void setUp() {
        template = HttpRequest.request("/v1/allFolders")
            .withMethod("GET");
        executor = Executors.newSingleThreadExecutor();
        identityClient = new HttpIdentityClient(
            IdentityClientOptions.forAddress("localhost", mockServerRule.getPort())
            .withHandlerExecutor(executor));
    }

    @After
    public void tearDown() throws Exception {
        executor.shutdown();
        executor.awaitTermination(5, TimeUnit.SECONDS);
    }

    @Test
    public void empty() {
        mockServerRule.getClient()
            .when(template.clone()
                .withQueryStringParameter("cloudId", "empty-cloud"))
            .respond(HttpResponse.response()
                .withBody(readResource("empty.json"), JSON_UTF_8));

        List<Folder> folders = identityClient.allFolders("empty-cloud").join();
        Assert.assertTrue(folders.isEmpty());
    }

    @Test
    public void nonEmpty() {
        // page 2
        mockServerRule.getClient()
            .when(template.clone()
                .withQueryStringParameter("cloudId", "my-cloud")
                .withQueryStringParameter("pageToken", "page2"))
            .respond(HttpResponse.response()
                .withBody(readResource("folders-page2.json"), JSON_UTF_8));

        // page 1 (XXX: must be defined after page2)
        mockServerRule.getClient()
            .when(template.clone()
                .withQueryStringParameter("cloudId", "my-cloud"))
            .respond(HttpResponse.response()
                .withBody(readResource("folders-page1.json"), JSON_UTF_8));

        List<Folder> folders = identityClient.allFolders("my-cloud").join();
        Assert.assertEquals(3, folders.size());
        Assert.assertEquals(new Folder("my-cloud", "folder1", "folder1-name"), folders.get(0));
        Assert.assertEquals(new Folder("my-cloud", "folder2", "folder2-name"), folders.get(1));
        Assert.assertEquals(new Folder("my-cloud", "folder3", "folder3-name"), folders.get(2));
    }

    @Test
    public void unknownFolder() {
        boolean gotExpectedException = false;
        try {
            identityClient.allFolders("1234").join();
        } catch (Exception e) {
            gotExpectedException = e.getMessage().contains("non OK response: 404");
        }
        Assert.assertTrue(gotExpectedException);
    }

    private String readResource(String name) {
        try {
            return Resources.toString(getClass().getResource(name), StandardCharsets.UTF_8);
        } catch (IOException e) {
            throw new UncheckedIOException(e);
        }
    }
}
